# -*- coding: utf-8 -*-
import six
import base64
import copy
import getpass
import json
import logging
import re
import ssl
import stat
import subprocess
import sys
import termios
import time
import tty
from ctypes import c_int, c_char_p, CDLL

import os
import syslog

from cinder.backup.lib import cpsrestful as cpscli
from cinder.backup.drivers.ebackupagent import Connection
from cinder.backup.lib.ebackuprestful import EbackupConnectionClient
from six.moves import configparser as config_parser
if six.PY3:
    import urllib.request as url_lib
else:
    import urllib2 as url_lib


def is_dj_env():
    if os.path.exists(DJ_ENV_FILE):
        return True
    return False


try:
    try:
        from FSSecurity import crypt as fscrypt
    except ImportError:
        from FSComponentUtil import crypt as fscrypt
except ImportError as msg:
    if not is_dj_env():
        raise Exception(msg)


logger = logging.getLogger('__EBACKUP_CONFIG__')
logger.setLevel(level=logging.DEBUG)
logger.propagate = False
logger.handlers = []

handler = logging.FileHandler('/var/log/ebackup_install_config.log')
handler.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s [%(levelname)s] at %(filename)s,%(lineno)d: %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)
handler = logging.StreamHandler(sys.stdout)
handler.setLevel(logging.INFO)
formatter = logging.Formatter('%(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)
DOCKING_USER = 4
AK_MAX_LENGTH = 256
SK_MAX_LENGTH = 256
PORT_LIMIT = 65535
LENGTH_LIMIT_32 = 32
LENGTH_LIMIT_255 = 255
CHOOSE_EXIT = '0'
CHOOSE_SHOW = '1'
CHOOSE_EBACKUP_SERVER_INFO = '2'
CHOOSE_BACKUP_ENV_INFO = '3'
CHOOSE_GLOBAL_CONFIGURATION = '4'
CHOOSE_PUBLIC_CLOUD = '1'
CHOOSE_PRIVATE_CLOUD = '2'
CHOOSE_HYBRID_CLOUD = '3'

CHOOSE_ADD_NEW_EBACKUP = '1'
CHOOSE_UPDATE_EBACKUP = '2'
CHOOSE_DELETE_EBACKUP = '3'
CHOOSE_RETUEN_EBACKUP_MENU = '0'

ENV_PUBLIC_CLOUD = '1'
ENV_PRIVATE_CLOUD = '2'
ENV_HYBRID_CLOUD = '3'
UNIT_UDS = '1'
UNIT_NAS = '2'
UNIT_SAN = '3'
DEFAULT_PORT = '8088'
DJ_ENV_FILE = '/etc/huawei/dj/cfg/sys.ini'
PROTOCOL_HTTP = '0'
PROTOCOL_HTTPS = '1'
DATALAYOUT_COMMON_FILESYSTEM = '0'
DATALAYOUT_DEDUPE = '1'
DATALAYOUT_COMPRESS = '2'
DATALAYOUT_COMPRESS_DEDUPE = '3'
DATALAYOUT_COMPRESS_DISTRIBUTE_DEDUPE = '6'
DATALAYOUT_DEFAULT = 255
TASK_RETRY_COUNT = 90
TASK_RETRY_TIME_INTERVAL = 30
UPGRADE_FILE = '/tmp/upgrade/driver_upgrade.conf'
UPGRADE_PATH = '/tmp/upgrade'

DEFAULT_DJ_SECTION = 'config'
DEFAULT_CPS_SECTION = 'default'

EBACKUP_SERVER_IP = "ebackup_server_ip"
EBACKUP_SERVER_PORT = "ebackup_server_port"
EBACKUP_SERVER_USERNAME = "ebackup_login_username"
EBACKUP_SERVER_PASSWORD_V2 = "ebackup_login_password_v2"
EBACKUP_LOGIN_CRT = "ebackup_login_crt"
AZS = "azs"
FUSION_STORAGE_IP = "fusion_storage_ip"
PHYSICAL_NETWORK_TYPE = "physical_network_type"
FUSION_STORAGE_AGENT_IP = "fusion_storage_agent_ip"
VALID_VOLUME_DRIVER = ['cinder.volume.drivers.dsware.HuaweiDswareDriver',
                       'cinder.volume.drivers.huawei.vrm.vrm_driver.VRMDriver',
                       'cinder.volume.drivers.huawei.HuaweiISCSIDriver',
                       'cinder.volume.drivers.dsware.HuaweiDswareIscsiDriver']
DEFAULT_ATTR = ['ebackup_uds_data_layout', 'ebackup_nfs_data_layout', 'ebackup_cifs_data_layout',
                'ebackup_san_data_layout', 'ebackup_storage_unit_path', 'ebackup_bucket_name',
                'ebackup_storage_username_v2',
                'ebackup_storage_password_v2', 'ebackup_storage_s3_protocol', 'ebackup_env_type',
                'ebackup_task_retry_count', 'ebackup_task_retry_time_interval', 'backup_service_outside',
                'api_gateway_host',
                'api_gateway_ca_file', 'ebackup_servers', 'ebackup_lazyloading', 'backup_driver',
                'ebackup_nova_endpoint', 'ebackup_nova_ecs_api']
MOVE_TO_EBK_ATTR = [EBACKUP_SERVER_IP, EBACKUP_SERVER_PORT, EBACKUP_SERVER_USERNAME, EBACKUP_SERVER_PASSWORD_V2,
                    EBACKUP_LOGIN_CRT]
EBK_ATTR = MOVE_TO_EBK_ATTR + [AZS, FUSION_STORAGE_IP, PHYSICAL_NETWORK_TYPE, FUSION_STORAGE_AGENT_IP, 'uuid', 'extend',
                               'tenant_white_list']
TEMPLATE_ATTR = ['volume_drivers', 'UnUseCinderBackup', 'templatenames', 'valid_volume_driver_names']

DMK_RET_SUCESS = 0
DMK_RET_FAILED = 1

ip_v4_compile = '((?:(?:25[0-5]|2[0-4]\d|((1\d{2})|([1-9]?\d)))\.){3}(?:25[0-5]|2[0-4]\d|((1\d{2})|([1-9]?\d))))$'
ip_v6_compile = '((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})' \
                '|(([0-9A-Fa-f]{1,4}:){5}(:[0-9A-Fa-f]{1,4}){1,2})|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]' \
                '{1,4}){1,3})|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){1,4})|(([0-9A-Fa-f]{1,4}:){2}' \
                '(:[0-9A-Fa-f]{1,4}){1,5})|(([0-9A-Fa-f]{1,4})?:(:[0-9A-Fa-f]{1,4}){1,6})|(([0-9A-Fa-f]{1,4}:)' \
                '{1,6}:)|(::)|(([0-9A-Fa-f]{1,4}:){6}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)' \
                '(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){5}:' \
                '(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]' \
                '|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,1}(25[0-5]|2[0-4][0-9]' \
                '|1[0-9][0-9]|[1-9][0-9]?|0)(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|' \
                '(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,2}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)' \
                '(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4}:){2}:' \
                '([0-9A-Fa-f]{1,4}:){0,3}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)' \
                '(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3})|(([0-9A-Fa-f]{1,4})?::' \
                '([0-9A-Fa-f]{1,4}:){0,4}(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)' \
                '(\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]?|0)){3}))'

g_template_names = []


class ebackup_server_config(object):
    ebackup_server_ip = ''
    ebackup_server_port = ''
    ebackup_login_username = ''
    ebackup_login_password = ''
    ebackup_login_crt = ''


class check_param:
    # the method name must be begin with 'check_param',then combine with [key]
    @staticmethod
    def check_param_backup_driver(value):
        if value not in ['cinder.backup.drivers.hwsebackup', 'cinder.backup.drivers.swift']:
            return False
        return True

    @staticmethod
    def check_param_ebackup_uds_data_layout(value):
        if value not in [DATALAYOUT_COMMON_FILESYSTEM, DATALAYOUT_COMPRESS]:
            return False
        return True

    @staticmethod
    def check_param_ebackup_nfs_data_layout(value):
        if value not in [DATALAYOUT_COMMON_FILESYSTEM, DATALAYOUT_DEDUPE, DATALAYOUT_COMPRESS,
                         DATALAYOUT_COMPRESS_DEDUPE]:
            return False
        return True

    @staticmethod
    def check_param_ebackup_cifs_data_layout(value):
        if value not in [DATALAYOUT_COMMON_FILESYSTEM, DATALAYOUT_COMPRESS]:
            return False
        return True

    @staticmethod
    def check_param_ebackup_san_data_layout(value):
        if value not in [DATALAYOUT_COMMON_FILESYSTEM, DATALAYOUT_DEDUPE, DATALAYOUT_COMPRESS,
                         DATALAYOUT_COMPRESS_DEDUPE]:
            return False
        return True

    @staticmethod
    def check_param_ebackup_storage_unit_path(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_bucket_name(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_storage_username_v2(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_storage_password_v2(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_storage_s3_protocol(value):
        if value not in [PROTOCOL_HTTP, PROTOCOL_HTTPS]:  # 0:
            return False
        return True

    @staticmethod
    def check_param_ebackup_env_type(value):
        if value not in [ENV_PUBLIC_CLOUD, ENV_PRIVATE_CLOUD, ENV_HYBRID_CLOUD]:
            return False
        return True

    @staticmethod
    def check_param_ebackup_task_retry_count(value):
        if int(value) < 0 or int(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_task_retry_time_interval(value):
        if int(value) < 0 or int(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_backup_service_outside(value):
        if value not in ['True', 'False', 'default']:
            return False
        return True

    @staticmethod
    def check_param_api_gateway_host(value):
        if not value.startswith('https://') or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_api_gateway_ca_file(value):
        if not os.path.exists(value):
            return False
        return True

    @staticmethod
    def check_param_ebackup_servers(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_lazyloading(value):
        # 0 means not support, 1 means support, 2 means support and dose not check op_gated_bklld
        if value not in ['0', '1', '2', '3', '4']:
            return False
        return True

    @staticmethod
    def check_param_azs(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_server_ip(value):
        if not check_string(value, ip_v4_compile) and not check_string(value, ip_v6_compile):
            return False
        return True

    @staticmethod
    def check_param_ebackup_server_port(value):
        if not value.isdigit() or int(value) > PORT_LIMIT:
            return False
        return True

    @staticmethod
    def check_param_ebackup_login_username(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_32:
            return False
        return True

    @staticmethod
    def check_param_ebackup_login_password_v2(value):
        if len(value) < 0 or len(value) > LENGTH_LIMIT_255:
            return False
        return True

    @staticmethod
    def check_param_ebackup_login_crt(value):
        if not os.path.exists(value):
            return False
        return True

    @staticmethod
    def check_param_fusion_storage_ip(value):
        if value == 'default':
            return True
        if not check_string(value, ip_v4_compile) and not check_string(value, ip_v6_compile):
            return False
        return True

    @staticmethod
    def check_param_physical_network_type(value):
        if value not in ['ip', 'ib']:
            return False
        return True

    @staticmethod
    def check_param_fusion_storage_agent_ip(value):
        agent_ips = value.split(',')
        if len(agent_ips) > 3:
            return False
        for ip in agent_ips:
            if not check_string(ip, ip_v4_compile) and not check_string(ip, ip_v6_compile):
                return False
        return True

    @staticmethod
    def check_param_ebackup_nova_endpoint(value):
        if len(value) < 1024 and value.startswith('http'):
            return True
        else:
            return False

    @staticmethod
    def check_param_ebackup_nova_ecs_api(value):
        if len(value) < 1024 and value.startswith('http'):
            return True
        else:
            return False

    @staticmethod
    def check_param_templatenames(value):
        if len(value) < 1024 and value.find('backup') > 1:
            return True
        else:
            return False

    @staticmethod
    def check_param_valid_volume_driver_names(value):
        if len(value) < 2048 and value in VALID_VOLUME_DRIVER:
            return True
        else:
            return False

    @staticmethod
    def check_param_uuid(value):
        return True

    @staticmethod
    def check_param_extend(value):
        return True

    @staticmethod
    def check_param_tenant_white_list(value):
        return True


def ebackupconfig_raw_input(input_str):
    return six.moves.input(input_str)


def load_kmc(kmc_path):
    _KMC_LIB_PATH = os.path.join(str(kmc_path), "lib", "libkmcadaptor.so")
    kmc = CDLL(_KMC_LIB_PATH)
    kmc.aesEncrypt.argtypes = [c_int, c_char_p]
    kmc.aesDecrypt.argtypes = [c_int, c_char_p]
    kmc.aesInit.argtypes = [c_char_p]
    kmc.aesInit.restype = c_int
    kmc.aesEncrypt.restype = c_char_p
    kmc.aesDecrypt.restype = c_char_p
    data_path = "/home/openstack/"
    kmc.aesInit(data_path)
    return kmc


if sys.version_info >= (2, 7, 9):
    ssl._create_default_https_context = ssl._create_unverified_context


def input_config_path():
    # input install path  and  config file path
    while True:
        config_file_path = six.moves.input(
            'Please enter the path of the cinder backup configuration file(default: /etc/cinder/):\n')
        config_file_path = config_file_path.strip()
        if config_file_path == '':
            config_file_path = '/etc/cinder/'
            break
        # check if abs path 

        if not config_file_path.startswith('/'):
            logger.info('The entered path must be an absolute path. Please check.')
            continue

        if not config_file_path.endswith('/'):
            config_file_path = config_file_path + '/'

        if os.path.isdir(config_file_path):
            break

        logger.info('Incorrect path. Please check it.')
    return config_file_path


def input_install_path():
    while True:
        cinder_install_path = six.moves.input(
            'Please enter the path of the eBackup driver configuration script(default: /usr/lib/python2.7/site-packages/cinder/backup/):\n')
        cinder_install_path = cinder_install_path.strip()
        if cinder_install_path == '':
            cinder_install_path = '/usr/lib/python2.7/site-packages/cinder/backup/'
            break
        if not cinder_install_path.startswith('/'):
            logger.info('The entered path must be an absolute path. Please check.')
            continue
        if not cinder_install_path.endswith('/'):
            cinder_install_path = cinder_install_path + '/'

        if os.path.isdir(cinder_install_path):
            break
        # if cinder manager is not exist, the path is wrong.
        logger.info('Incorrect path. Please check it.')
    return cinder_install_path


def get_install_path():
    default_install_path = ''
    if os.path.isdir('/usr/lib/python3.7/site-packages/cinder/backup/'):
        default_install_path = \
            '/usr/lib/python3.7/site-packages/cinder/backup/'
    elif os.path.isdir('/usr/lib/python2.7/site-packages/cinder/backup/'):
        default_install_path = '/usr/lib/python2.7/site-packages/cinder/backup/'
    elif os.path.isdir('/usr/lib64/python2.7/site-packages/cinder/backup/'):
        default_install_path = '/usr/lib64/python2.7/site-packages/cinder/backup/'
    elif os.path.isdir('/opt/cloud/services/cascading-cinder/venv/lib/python2.7/site-packages/cinder/backup/'):
        default_install_path = '/opt/cloud/services/cascading-cinder/venv/lib/python2.7/site-packages/cinder/backup/'
    else:
        logger.info('Can not get default path.')

    while True:
        cinder_install_path = six.moves.input(
            'Please enter the path of the eBackup driver configuration script(default: %s):\n' % default_install_path)
        cinder_install_path = cinder_install_path.strip()
        if cinder_install_path == '':
            if default_install_path != '':
                cinder_install_path = default_install_path
                break
            else:
                continue
        if not cinder_install_path.startswith('/'):
            logger.info('The entered path must be an absolute path. Please check.')
            continue
        if not cinder_install_path.endswith('/'):
            cinder_install_path = cinder_install_path + '/'

        if os.path.isdir(cinder_install_path):
            break
        # if cinder manager is not exist, the path is wrong.
        logger.info('Incorrect path. Please check it.')
    return cinder_install_path


def get_install_path_slient():
    install_path = ''
    if os.path.isfile('/tmp/ebackup_driver_path'):
        with open('/tmp/ebackup_driver_path') as fh:
            line = fh.readline()
            install_path = line.strip()
    elif os.path.isdir('/usr/lib/python3.7/site-packages/cinder/backup/'):
        install_path = '/usr/lib/python3.7/site-packages/cinder/backup/'
    elif os.path.isdir('/usr/lib/python2.7/site-packages/cinder/backup/'):
        install_path = '/usr/lib/python2.7/site-packages/cinder/backup/'
    elif os.path.isdir('/usr/lib64/python2.7/site-packages/cinder/backup/'):
        install_path = '/usr/lib64/python2.7/site-packages/cinder/backup/'
    elif os.path.isdir('/opt/cloud/services/cascading-cinder/venv/lib/python2.7/site-packages/cinder/backup/'):
        install_path = '/opt/cloud/services/cascading-cinder/venv/lib/python2.7/site-packages/cinder/backup/'
    else:
        logger.info('Can not get default path.')
        raise Exception('Can not get default path.')
    if not os.path.isdir(install_path):
        raise Exception('Invalid install path.')
    return install_path


def set_log(msgs):
    syslog.openlog('ebackupdriver', syslog.LOG_PID, syslog.LOG_AUTH)
    syslog.syslog(msgs)


def check_ebackup_pwd(ebackup_ip, ebackup_port, ebackup_username, ebackup_password):
    if check_string(ebackup_ip, ip_v6_compile):
        login_uri = ''.join(['https://[', ebackup_ip, ']:', ebackup_port, '/rest/dev/sessions'])
    else:
        login_uri = ''.join(['https://', ebackup_ip, ':', ebackup_port, '/rest/dev/sessions'])
    login_body = {
        "username": ebackup_username,
        "password": ebackup_password,
        "scope": DOCKING_USER
    }

    logger.info('Checking the user name and password. Please wait...')
    try:
        proxy = url_lib.ProxyHandler({})
        opener = url_lib.build_opener(proxy)
        url_lib.install_opener(opener)
        jdata = json.dumps(login_body)
        if six.PY3:
            jdata = jdata.encode('utf-8')
        req = url_lib.Request(login_uri, jdata)
        response = url_lib.urlopen(req, timeout=30)
        result = json.loads(response.read())
        errordata = result['error']
        errornum = errordata['code']
        if errornum != 0:
            err_str = "Check failed, error number: " + str(errornum)
            if 'description' in errordata.keys():
                err_str = err_str + ", reason: " + errordata['description']
            logger.info(err_str)
            return False
    except Exception as msg:
        logger.info('login eBackup failed, please check eBackup server {}.'.format(msg))
        return False
    return True


def check_string(src_string, compile_re):
    re_com = re.compile(compile_re)
    if re_com.match(src_string):
        return True
    else:
        return False


def get_section_name_with_ip(ipaddr):
    if check_string(ipaddr, ip_v6_compile):
        nums = re.split("[:.]", ipaddr.lower())
    else:
        nums = ipaddr.split(".")
        for i in range(len(nums)):
            hexnum = hex(int(nums[i]))
            hexstr = str(hexnum)
            nums[i] = hexstr[2:]
    return 'ebk_' + ''.join(nums)


def print_ebackup_server_config(server_item):
    if server_item.get(EBACKUP_SERVER_IP, "") != "":
        logger.info("# eBackup Server Management IP Address : {}".format(server_item.get(EBACKUP_SERVER_IP)))
    if server_item.get(EBACKUP_SERVER_PORT, "") != "":
        logger.info("# eBackup Server Port : {}".format(server_item.get(EBACKUP_SERVER_PORT)))
    if server_item.get(EBACKUP_SERVER_USERNAME, "") != "":
        logger.info("# eBackup Server Login Name : {}".format(server_item.get(EBACKUP_SERVER_USERNAME)))
    if server_item.get(EBACKUP_LOGIN_CRT, "") != "":
        logger.info("# eBackup Server Login Certification File : {}".format(server_item.get(EBACKUP_LOGIN_CRT)))
    if server_item.get(AZS, "") != "":
        logger.info("# eBackup Server Available Zone : {}".format(server_item.get(AZS)))
    if server_item.get(FUSION_STORAGE_IP, "") != "":
        logger.info("# eBackup Server Fusion Storage IP : {}".format(server_item.get(FUSION_STORAGE_IP)))
    if server_item.get(FUSION_STORAGE_AGENT_IP, "") != "":
        logger.info("# eBackup Server Fusion Storage Agent IP : {}".format(server_item.get(FUSION_STORAGE_AGENT_IP)))
    if server_item.get(PHYSICAL_NETWORK_TYPE, "") != "":
        logger.info("# eBackup Physical network type : {}".format(server_item.get(PHYSICAL_NETWORK_TYPE)))
        logger.info("-------------------------------------------------------------------------")


def _input_ak_sk():
    while True:
        ak = six.moves.input('Please enter the S3 access key:\n')
        if (ak.strip() == '') or (len(ak) > AK_MAX_LENGTH):  # AK MAX LENGTH
            logger.info('Invalid input. Please check it.')
            continue
        logger.info('Please enter the S3 secret key:')
        sk = get_asterisk_password('*')
        if (sk.strip() == '') or (len(sk) > SK_MAX_LENGTH):  # SK MAX LENGTH
            logger.info('Invalid input. Please check it.')
            continue
        break
    return ak, sk


def _input_uds_data_layout():
    while True:
        uds_layout_type = six.moves.input(
            'Please enter the data layout for S3:(0:common filesystem, 2:compress. default: 2):\n')
        if uds_layout_type.strip() == '':
            uds_layout_type = '2'
        if str(uds_layout_type) not in ['0', '2']:
            logger.info('Invalid input. Please check it.')
            continue
        return uds_layout_type


def _input_nas_data_layout():
    while True:
        nfs_layout_type = six.moves.input(
            'Please enter the data layout for NFS:(0:common filesystem, 1:dedupe 2:compress 3:compress and dedupe. default: 2):\n')
        if nfs_layout_type.strip() == '':
            nfs_layout_type = '2'
        if str(nfs_layout_type) not in ['1', '0', '2', '3']:
            logger.info('Invalid input. Please check it.')
            continue
        if str(nfs_layout_type) in ['1', '3']:
            choose = six.moves.input(
                "If Current license not support features about dedupe, this config will make backup failed, are you sure to change?[Y/N](default:n):")
            if choose not in ['Y', 'y']:
                continue
        cifs_layout_type = six.moves.input(
            'Please enter the data layout for CIFS:(0:common filesystem, 2:compress. default: 2):\n')
        if cifs_layout_type.strip() == '':
            cifs_layout_type = '2'
        if str(cifs_layout_type) not in ['0', '2']:
            logger.info('Invalid input. Please check it.')
            continue
        return nfs_layout_type, cifs_layout_type


def check_ebackup_certs(ebackup_ip, ebackup_port, cacrtf):
    try:
        if os.path.exists(cacrtf):
            ssl.get_server_certificate((ebackup_ip, int(ebackup_port)), ssl_version=ssl.PROTOCOL_TLSv1_2,
                                       ca_certs=cacrtf)
            logger.info('certificate verified success')
        else:
            logger.info('certificate error: file is not exist.')
            return False
    except Exception as msg:
        logger.info('warning:certificate invalid(%s)' % msg)
        while True:
            choose = six.moves.input(
                'Verifying the certificate failed. Continue to configure the certificate?(y/n)(default:y)'
                "\n"'Warning:If you choose n, security risks exist, but you can still use the certificate:')
            if choose == '' or choose == 'Y' or choose == 'y' or choose == 'N' or choose == 'n':
                break
        if choose == '' or choose == 'Y' or choose == 'y':
            return False
    return True


def _input_cert_path(ipaddr, port, default_path):
    while True:
        if default_path == '':
            default_path = '/usr/lib/python2.7/site-packages/cinder/backup/cacert.pem'
        attention = ("(default:" + default_path + ")")
        crt_path = ebackupconfig_raw_input('Please enter the certificate file path' + attention + ':\n')
        crt_path = crt_path.strip()
        if crt_path == '':
            crt_path = default_path
        if not crt_path.startswith('/'):
            logger.info('The entered path must be an absolute path. Please check.')
            continue
        certs_be_right = check_ebackup_certs(ipaddr, port, crt_path)
        if certs_be_right:
            break
    return crt_path


def _get_azs_from_nova():
    cmd = (
        "nova availability-zone-list|awk -F'|' '{print $2}'| sed '/^[ ]*$/d'|grep -E -v 'Name|internal'|sed 's/ //g'")
    pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout
    az_name_list = pipe.read()
    if six.PY3:
        az_name_list = str(az_name_list, encoding='utf-8')
    az_names = az_name_list.split("\n")
    msg = ("got az: %s" % az_names)
    logger.debug(msg)
    return az_names


def _get_input_azs_name(default_az):
    az_names = _get_azs_from_nova()
    attention = ''
    if default_az != '':
        attention = ("(default:" + default_az + ")")
    while True:
        azs_info = 'default'
        input_req = (
            'Please enter the availability zone (AZ) numbers' + attention + ', and isolate the numbers using commas (,), for example: 1,2:\n')
        error_input = False
        az_count = 0
        for i, az in enumerate(az_names):
            if az.strip() != '':
                input_req = '%s%d:%s\n' % (input_req, i + 1, az)
                az_count = az_count + 1
        az_numbers = ebackupconfig_raw_input(input_req)
        if az_numbers.strip() == '' and default_az != '':
            azs_info = default_az
            break
        az_number_array = az_numbers.split(",")
        for _number in az_number_array:
            if not _number.isdigit() or int(_number) > az_count or int(_number) <= 0:
                logger.info('Invalid input. Please check it.')
                error_input = True
                break
            else:
                if azs_info == 'default':
                    azs_info = az_names[int(_number) - 1]
                else:
                    azs_info = '%s,%s' % (azs_info, az_names[int(_number) - 1])
        if not error_input:
            break
    return azs_info


def _input_muti_cluster_extend_info():
    while True:
        extend_info = False
        tenant_write_list = ""
        azs_bool = ebackupconfig_raw_input(
            'Are the physical network type and AZ information of the eBackup server '
            'to be added the same as those of the existing eBackup server?(Y/N)\n')
        if azs_bool.lower().strip() == 'y' or azs_bool.lower().strip() == 'yes':
            extend_info = True
            tenant_write_list = ebackupconfig_raw_input(
                'Enter a whitelist containing the IDs of the tenants '
                'that are allowed to access the eBackup server. '
                'Use commas (,) to separate multiple tenant IDs, for example: ID1,ID2\n')
        elif azs_bool.lower().strip() == 'n' or azs_bool.lower().strip() == 'no':
            extend_info = False
        else:
            logger.info('Invalid input.')
            continue
        break
    return extend_info, tenant_write_list


def _input_azs_info(default_az):
    while True:
        azs_info = 'default'
        azs_bool = ebackupconfig_raw_input(
            'Configure availability zone (AZ) information for the eBackup server now?(Y/N)\n')
        if azs_bool.lower().strip() == 'y' or azs_bool.lower().strip() == 'yes':
            azs_info = _get_input_azs_name(default_az)
            break
        elif azs_bool.lower().strip() == 'n' or azs_bool.lower().strip() == 'no':
            break
        logger.info('Invalid input.')
    return azs_info


def _input_fusion_storage_agent_ip(default_agent_ip):
    format_ip = ''
    while True:
        if default_agent_ip:
            attention = "(default:" + default_agent_ip + ")"
        else:
            attention = "(Use commas to separate the IP addresses and a " \
                        "maximum of three IP addresses are supported.)"
        agent_ip = ebackupconfig_raw_input('Enter the Huawei Distributed '
                                           'Block Storage Agent IP '
                                           'address{}:\n'.format(attention))
        agent_ip = agent_ip.strip()

        if (agent_ip != '') and (len(agent_ip) > LENGTH_LIMIT_32 * 3 + 2):
            logger.info('Invalid input. Please check it.')
            continue
        elif agent_ip == '':
            format_ip = default_agent_ip
            break
        else:
            agent_ips = agent_ip.split(',')
            is_format = True
            if len(agent_ips) > 3:
                logger.info('the number of ip address has outnumber 3. Please check it.')
                continue
            for ip in agent_ips:
                ip = ip.strip(' ')
                if not check_string(ip, ip_v4_compile) and not check_string(ip, ip_v6_compile):
                    logger.info('Invalid input. Please check it.')
                    is_format = False
                    break
                else:
                    format_ip = format_ip + ip + ','
            if format_ip.endswith(','):
                format_ip = format_ip[0:len(format_ip) - 1]
            if is_format:
                break

    return format_ip


def _get_fusion_storage_info(default_fm, default_pht, default_agent_ip):
    pht = ''
    _agent_ipaddr = ''
    while True:
        attention = ''
        if default_fm:
            attention = "(default:" + default_fm + ")"
        _ipaddr = ebackupconfig_raw_input('Enter the IP address of FusionStorage' + attention + ":\n")
        _ipaddr = _ipaddr.strip()

        if (_ipaddr != '') and (len(_ipaddr) > LENGTH_LIMIT_255):
            logger.info('Invalid input. Please check it.')
            continue
        elif _ipaddr == '':
            if default_fm:
                _ipaddr = default_fm
                break
            else:
                logger.info('Invalid input. Please check it.')
                continue
        else:
            if _ipaddr != 'default' and not check_string(_ipaddr, ip_v4_compile) and not check_string(_ipaddr,
                                                                                                      ip_v6_compile):
                logger.info('Invalid input. Please check it.')
                continue
            break

    while True:
        fa_bool = ebackupconfig_raw_input('Do you want to configure Huawei '
                                          'Distributed Block Storage Agent IP '
                                          'now?(Y/N)\n')
        if fa_bool.lower().strip() == 'y' or fa_bool.lower().strip() == 'yes':
            _agent_ipaddr = _input_fusion_storage_agent_ip(default_agent_ip)
            break
        elif fa_bool.lower().strip() == 'n' or fa_bool.lower().strip() == 'no':
            break
        else:
            logger.info('Invalid input.')

    if _ipaddr == 'default':
        return _ipaddr, pht, _agent_ipaddr

    while True:
        attention = ''
        if default_pht:
            attention = "(default:" + default_pht + ")"
        pht = ebackupconfig_raw_input('Enter the physical network type of FusionStorage(option:ib, ip or press enter)'
                                      + attention + ":\n")
        pht = pht.strip()
        if pht != 'ip' and pht != 'ib' and pht != '':
            logger.info('Invalid input. Please input ip or ib')
            continue
        elif pht == '':
            if default_pht:
                pht = default_pht
                break
            else:
                logger.info('Nothing input, disable it')
                break
        else:
            break

    return _ipaddr, pht, _agent_ipaddr


def _input_fusion_storage_info(default_fm, default_pht, default_agent_ip):
    while True:
        _ipaddr = ''
        pht = ''
        _agent_ipaddr = ''
        fm_bool = ebackupconfig_raw_input('Do you want to configure Huawei '
                                          'Distributed Block Storage '
                                          'information for the eBackup server '
                                          'now?(Y/N)\n')
        if fm_bool.lower().strip() == 'y' or fm_bool.lower().strip() == 'yes':
            _ipaddr, pht, _agent_ipaddr = _get_fusion_storage_info(default_fm, default_pht, default_agent_ip)
            break
        elif fm_bool.lower().strip() == 'n' or fm_bool.lower().strip() == 'no':
            break
        logger.info('Invalid input.')
    return _ipaddr, pht, _agent_ipaddr


def _input_ebackup_info(default_conf):
    retry_count = 0
    attention = ''
    while retry_count < 3:
        if default_conf.ebackup_server_ip != '':
            attention = ("(default:" + default_conf.ebackup_server_ip + ")")
        _ipaddr = ebackupconfig_raw_input(
            'Please enter the IP address of the eBackup management plane' + attention + ':\n')
        _ipaddr = _ipaddr.lower().strip()
        attention = ''
        if ((_ipaddr == '') and default_conf.ebackup_server_ip == '') or (len(_ipaddr) > LENGTH_LIMIT_255):
            logger.info('Invalid input. Please check it.')
            continue
        elif _ipaddr == '':
            _ipaddr = default_conf.ebackup_server_ip

        if default_conf.ebackup_server_port == '':
            default_conf.ebackup_server_port = DEFAULT_PORT
        attention = ("(default:" + default_conf.ebackup_server_port + ")")
        _port = ebackupconfig_raw_input(
            'Please enter the port number of the eBackup management plane' + attention + ':\n')
        _port = _port.strip()
        attention = ''
        if _port == '':
            _port = default_conf.ebackup_server_port
        if not _port.isdigit() or int(_port) > PORT_LIMIT:
            logger.info('Invalid port. Port number ranges from 1024 to 65535.')
            continue

        if default_conf.ebackup_login_username != '':
            attention = ("(default:" + default_conf.ebackup_login_username + ")")
        _ebacup_user = ebackupconfig_raw_input(
            'Please enter the interface interconnection user name created in the eBackup management system' + attention + ':\n')
        _ebacup_user = _ebacup_user.strip()
        if ((_ebacup_user == '') and default_conf.ebackup_login_username == '') or (
                    len(_ebacup_user) > LENGTH_LIMIT_32):
            logger.info('Invalid input. Please check it.')
            continue
        elif _ebacup_user == '':
            _ebacup_user = default_conf.ebackup_login_username

        logger.info('Please enter the password of the interface interconnection user:')
        _ebackup_pwd = get_asterisk_password('*')
        if (_ebackup_pwd.strip() == '') or (len(_ebackup_pwd) > LENGTH_LIMIT_32):
            logger.info('Invalid input. Please check it.')
            continue
        be_right = check_ebackup_pwd(_ipaddr, _port, _ebacup_user, _ebackup_pwd)
        if be_right:
            break
        retry_count = retry_count + 1
    if retry_count >= 3:
        sys.exit()
    return _ipaddr, _port, _ebacup_user, _ebackup_pwd


def get_cinder_modes_by_conf(path):
    info_path = path + '/components_info'
    template_names = []
    if os.path.exists(info_path):
        for root, dirs, files in os.walk(info_path):
            for filed in files:
                if filed.startswith('cinder-backup'):
                    template_names.append((str(filed)).strip())
    return template_names


def get_cinder_conf_by_mode(path):
    info_path = path + 'components_info'
    if os.path.exists(info_path):
        for root, dirs, files in os.walk(info_path):
            for filed in files:
                if filed.startswith('cinder-backup-'):
                    return filed
    return 'cinder-backup'


def kill_process_by_name(name):
    cmd = "ps -e | grep %s" % name
    f = os.popen(cmd)
    txt = f.readlines()
    if len(txt) == 0:
        logger.info("process \"%s\" not exist" % name)
        return
    else:
        for line in txt:
            colum = line.split()
            pid = colum[0]
            cmd = "kill -9 %d" % int(pid)
            if not check_valid_char(cmd):
                logger.info("The params of cmd str are invalid, include illegal chars(| ; & $ > <).")
            rc = os.system(cmd)
            if rc == 0:
                logger.info("succeed to restart \"%s\"" % name)
            else:
                logger.info("failed to restart \"%s\"" % name)
    return


def check_service_is_running(template_name):
    if not check_valid_char(template_name):
        logger.info("The params of cmd str are invalid, include illegal chars(| ; & $ > <).")
    cmd = 'ps -ef | grep "cinder-backup" | grep "config-file" | grep "%s\\.conf" | grep -v "grep" > /dev/null 2>&1' \
          % str(template_name)
    retry_count = 0
    while retry_count < 3:
        rc = os.system(cmd)
        if rc == 0:
            return True
        retry_count = retry_count + 1
        time.sleep(5)
    return False


def cinder_restart_all(path):
    if is_dj_env():
        kill_process_by_name('cinder-backup')
        time.sleep(120)
    else:
        template_names = get_cinder_modes_by_conf(path)
        for template_name in template_names:
            service_is_running = check_service_is_running(template_name)
            if not service_is_running:
                continue
            cmd = '%sControl -A RESTART' % template_name
            if not check_valid_char(template_name):
                logger.info("The params of cmd str are invalid, include illegal chars(| ; & $ > <).")
            result = os.system(cmd)
            if result == 0:
                logger.info('succeed to restart %s' % template_name)


def add_conf_section(filepath, section_name):
    if is_dj_env():
        config = config_parser.ConfigParser()
        config.read(filepath)
        if config.has_section(section_name):
            return
        config.add_section(section_name)
        with open(filepath, 'w') as fwobj:
            config.write(fwobj)


def _print_ebackup_server(i, server_item):
    print_str = str(i) + ". serverip   : " + server_item.get(EBACKUP_SERVER_IP, "") + "\n"
    print_str += ("   port       : " + server_item.get(EBACKUP_SERVER_PORT, "") + "\n")
    print_str += ("   username   : " + server_item.get(EBACKUP_SERVER_USERNAME, "") + "\n")
    print_str += ("   certificate: " + server_item.get(EBACKUP_LOGIN_CRT, "") + "\n")
    print_str += ("   availability zone: " + server_item.get(AZS, "") + "\n")
    print_str += ("   fusion storage ip: " + server_item.get(FUSION_STORAGE_IP, "") + "\n")
    print_str += ("   fusion storage agent ip: " + server_item.get(FUSION_STORAGE_AGENT_IP, "") + "\n")
    print_str += ("   physical network type: " + server_item.get(PHYSICAL_NETWORK_TYPE, "") + "\n")
    return print_str


def change_kmc_keystore_permission(basepath):
    if not check_valid_char(basepath):
        logger.info("The params of cmd str are invalid, include illegal chars(| ; & $ > <).")
    primary_keystore_file = basepath + "primary_keystore.dat"
    standby_keystore_file = basepath + "standby_keystore.dat"
    primary_kmc_cfg_file = basepath + "primary_kmc_cfg.conf"
    standby_kmc_cfg_file = basepath + "standby_kmc_cfg.conf"
    os.system("chown openstack:openstack " + basepath + "*.dat")
    os.system("chown openstack:openstack " + basepath + "*.conf")
    os.chmod(primary_keystore_file, stat.S_IWRITE | stat.S_IREAD)
    os.chmod(standby_keystore_file, stat.S_IWRITE | stat.S_IREAD)
    os.chmod(primary_kmc_cfg_file, stat.S_IWRITE | stat.S_IREAD)
    os.chmod(standby_kmc_cfg_file, stat.S_IWRITE | stat.S_IREAD)


def set_env_var():
    logger.info('Please set the cps environment variables first.')
    while True:
        while True:
            cps_username = six.moves.input('Enter CPS_USERNAME=')
            if cps_username.strip() == '':
                logger.info("CPS_USERNAME can not be empty,please input again.")
            else:
                break
        os.environ['CPS_USERNAME'] = cps_username

        while True:
            cps_password = getpass.getpass('Enter CPS_PASSWORD=')
            if cps_password.strip() == '':
                logger.info("CPS_PASSWORD can not be empty,please input again.")
            else:
                break
        os.environ['CPS_PASSWORD'] = cps_password

        cmd = 'cps host-list'
        pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE).stdout
        cps_host_list_result = pipe.read()
        if six.PY3:
            cps_host_list_result = str(cps_host_list_result, encoding='utf-8')
        if cps_host_list_result.find('locked') != -1:
            logger.info("CPS user is current locked, please input again later.")
            sys.exit()
        elif cps_host_list_result.find('id') == -1 and cps_host_list_result.find('status') == -1:
            logger.info("CPS_USERNAME or CPS_PASSWORD is invalid, please input again.")
        else:
            break
    return cps_username, cps_password


def hwconfig_main():
    logger.info('configuration eBackup driver start.')
    file_abs_path = os.path.dirname(os.path.abspath(__file__))
    filepath = str(file_abs_path) + "/drivers/ebackup.conf"
    add_conf_section(filepath, "config")
    if is_dj_env():
        change_kmc_keystore_permission("/home/openstack/")

    if not is_dj_env():
        cps_user, cps_password = set_env_var()

    cfg = InteractiveConfig(cps_user, cps_password)
    cfg.init_and_get_parameters()
    cfg.begin_config()


class FileConfig(object):
    def __init__(self, init_file):
        self.parser = config_parser.ConfigParser()
        self.parser.read(init_file)
        self.cps_username = self.parser.get('CpsInfo', 'cpsusername').strip()
        os.environ['CPS_USERNAME'] = self.cps_username
        while True:
            self.cps_password = getpass.getpass('type the password of cps user(%s):' % self.cps_username)
            if not self.cps_username.strip() or not self.cps_password.strip():
                logger.info("cps_username and cps_password can not be empty,please input again.")
            else:
                os.environ['CPS_PASSWORD'] = self.cps_password
                pipe = subprocess.Popen('cps host-list', shell=True, stdout=subprocess.PIPE).stdout
                cps_host_list_result = pipe.read()
                if six.PY3:
                    cps_host_list_result = str(cps_host_list_result,
                                               encoding='utf-8')
                if cps_host_list_result.find('locked') != -1:
                    logger.info("cps user is current locked, please try again later.")
                    sys.exit()
                elif cps_host_list_result.find('id') == -1 and cps_host_list_result.find('status') == -1:
                    logger.info("cps username or cps password is invalid, please check out your cps_username "
                                "in ini file or input password again.")
                else:
                    break

    @staticmethod
    def _input_uds_ak_sk():
        while True:
            ak = six.moves.input('Please enter the S3 access key:\n')
            if (ak.strip() == '') or (len(ak) > AK_MAX_LENGTH):
                logger.info('Invalid input. Please check it.')
                continue
            logger.info('Please enter the S3 secret key:')
            sk = get_asterisk_password('*')
            if (sk.strip() == '') or (len(sk) > SK_MAX_LENGTH):
                logger.info('Invalid input. Please check it.')
                continue
            break
        return ak, sk

    @staticmethod
    def _input_ebackup_password(ipaddr, port, ebacup_user):
        retry_count = 0
        while retry_count < 3:
            logger.info('Please enter the password of the interface interconnection user:')
            ebackup_pwd = get_asterisk_password('*')
            if (ebackup_pwd.strip() == '') or (len(ebackup_pwd) > LENGTH_LIMIT_32):
                logger.info('Invalid input. Please check it.')
                continue
            be_right = check_ebackup_pwd(ipaddr, port, ebacup_user, ebackup_pwd)
            if not be_right:
                logger.info('Incorrect user name or password.')
                retry_count += 1
            else:
                return ebackup_pwd
        exit(1)

    def file_config_commit(self):
        paras = dict()
        env_type = self.parser.get('CloudSchemeInfo', 'cloudscheme').strip()
        ebk_section = get_section_name_with_ip(self.parser.get('ConnInfo', 'ipaddr')).strip()
        ebackup_conf = self.parser._sections['ConnInfo']
        paras[ebk_section + '.' + EBACKUP_SERVER_IP] = ebackup_conf.get('ipaddr').strip()
        paras[ebk_section + '.' + EBACKUP_SERVER_PORT] = ebackup_conf.get('port').strip()
        paras[ebk_section + '.' + EBACKUP_SERVER_USERNAME] = ebackup_conf.get('loginuser').strip()
        paras[ebk_section + '.' + EBACKUP_LOGIN_CRT] = ebackup_conf.get('certfilepath').strip()
        paras[ebk_section + '.' + AZS] = ebackup_conf.get('azs').strip()
        paras[ebk_section + '.' + FUSION_STORAGE_IP] = ebackup_conf.get('fusionstorageip').strip()
        paras[ebk_section + '.' + PHYSICAL_NETWORK_TYPE] = ebackup_conf.get('physicalnetworktype').strip()
        paras[ebk_section + '.' + FUSION_STORAGE_AGENT_IP] = ebackup_conf.get('fusionstorageagentip').strip()
        if env_type == 'public':
            paras['default.ebackup_env_type'] = ENV_PUBLIC_CLOUD
            public_conf = self.parser._sections['PublicBackupEnvInfo']
            paras['default.ebackup_storage_s3_protocol'] = public_conf.get('s3protocol').strip()
            paras['default.ebackup_storage_unit_path'] = public_conf.get('s3ipaddr').strip()
            paras['default.ebackup_bucket_name'] = public_conf.get('s3bucketprefix').strip()
            paras['default.ebackup_uds_data_layout'] = public_conf.get('s3datalayout').strip()
            paras['default.ebackup_lazyloading'] = public_conf.get('lazyloading').strip()
            paras['default.ebackup_nova_endpoint'] = public_conf.get('ebackup_nova_endpoint').strip()
            paras['default.ebackup_nova_ecs_api'] = public_conf.get('ebackup_nova_ecs_api').strip()
        elif env_type == 'private':
            paras['default.ebackup_env_type'] = ENV_PRIVATE_CLOUD
            private_conf = self.parser._sections['PrivateBackupEnvInfo']
            storage_type = private_conf.get('storageunittype').strip()
            if 'UDS' in storage_type:
                paras['default.ebackup_uds_data_layout'] = private_conf.get('s3datalayout').strip()
            if 'NAS' in storage_type:
                paras['default.ebackup_nfs_data_layout'] = private_conf.get('nfsdatalayout').strip()
                paras['default.ebackup_cifs_data_layout'] = private_conf.get('cifsdatalayout').strip()
        elif env_type == 'hybrid':
            paras['default.ebackup_env_type'] = ENV_HYBRID_CLOUD
            hybird_conf = self.parser._sections['HybridBackupEnvInfo']
            storage_type = hybird_conf.get('StorageUnitType').strip()
            if 'UDS' in storage_type:
                paras['default.ebackup_uds_data_layout'] = hybird_conf.get('s3datalayout').strip()
            if 'NAS' in storage_type:
                paras['default.ebackup_nfs_data_layout'] = hybird_conf.get('nfsdatalayout').strip()
                paras['default.ebackup_cifs_data_layout'] = hybird_conf.get('cifsdatalayout').strip()
            if 'SAN' in storage_type:
                paras['default.ebackup_san_data_layout'] = hybird_conf.get('sandatalayout').strip()
        else:
            logger.info('The cloud scheme is incorrect. Please enter a correct one (public, private, or hybrid).')
            sys.exit(1)
        if self.parser.get('BackupDataEncrypt', 'EnEncrypt') == 'True':
            paras['default.api_gateway_host'] = self.parser.get('BackupDataEncrypt', 'apigatewayhost').strip()
            paras['default.api_gateway_ca_file'] = self.parser.get('BackupDataEncrypt', 'apigatewaycafile').strip()

        paras_check = copy.deepcopy(paras)
        for longkey, value in paras_check.items():
            if not value:
                logger.debug("poped unassigned key:" + longkey)
                paras.pop(longkey)
                continue
            key = longkey.split('.')[1]
            if key in DEFAULT_ATTR or key in EBK_ATTR or key in TEMPLATE_ATTR:
                logger.debug("check param for key:" + key)
                func = getattr(check_param, "check_param_" + key)
                if not func(value):
                    logger.info(longkey + " check failed!")
                    sys.exit(1)
            else:
                logger.info(longkey + " not in attibute")
                sys.exit(1)
        paras[ebk_section + '.' + EBACKUP_SERVER_PASSWORD_V2] = self._input_ebackup_password(
            paras[ebk_section + '.' + EBACKUP_SERVER_IP], paras[ebk_section + '.' + EBACKUP_SERVER_PORT],
            paras[ebk_section + '.' + EBACKUP_SERVER_USERNAME])
        if env_type == 'public':
            paras['default.ebackup_storage_username_v2'], paras[
                'default.ebackup_storage_password_v2'] = self._input_uds_ak_sk()
        paras['default.backup_driver'] = 'cinder.backup.drivers.hwsebackup'
        paras['default.backup_service_outside'] = 'True'
        paras['default.ebackup_task_retry_count'] = '90'
        paras['default.ebackup_task_retry_time_interval'] = '30'
        paras['default.ebackup_servers'] = ebk_section
        cpslinker = CpsConfig('', '', self.cps_username, self.cps_password)
        cpslinker.set_backup_extend_parameters(paras, self.parser.get('UnUseCinderBackup', 'templatenames').split(),
                                               self.parser.get('volume_drivers', 'valid_volume_driver_names').split(),
                                               dispatch_params_to_all=True, full_cover=True)
        logger.info('commit all conf success')


class DmkConfig(object):
    def __init__(self, cps_name, cps_pwd_file='', cps_password=''):
        self.cps_name = cps_name
        self.cps_file = cps_pwd_file
        self.cps_pwd_value = cps_password
        if cps_pwd_file:
            cps_pwd_key, self.cps_pwd_value = self.get_pwd_from_file(self.cps_file)
            if cps_pwd_key != 'cps_userpasswd':
                logger.info('%s not has cps_userpasswd' % self.cps_file)
                sys.exit(1)

    @staticmethod
    def check_param_for_dmk(key, value):
        if key in DEFAULT_ATTR or key in EBK_ATTR or key in TEMPLATE_ATTR:
            logger.info("check param for key:" + key)
            # method check_param_XXX need exist with key
            func = getattr(check_param, "check_param_" + key)
            return func(value)
        logger.info(key + " not in attribute")
        sys.exit(1)

    @staticmethod
    def write_conf_to_file(key, value, section, filename):
        logger.info("write conf to file start.")
        if is_dj_env():
            logger.info("this is dj env, no need rewrite config.")
            return
        if (section == 'default' and key in DEFAULT_ATTR) or (section.startswith('ebk') and key in EBK_ATTR) or \
                (section in TEMPLATE_ATTR):
            if not os.path.isfile(filename):
                os.mknod(filename)
            try:
                cfg = config_parser.ConfigParser()
                cfg.read(filename)
            except IOError:
                logger.error("open " + filename + " failed.")
                sys.exit(1)
            if six.PY3 and isinstance(value, bytes):
                value = str(value, encoding="utf-8")
            if section == 'default':
                cfg.set(config_parser.DEFAULTSECT, key, value)
            else:
                if section not in cfg.sections():
                    cfg.add_section(section)
                cfg.set(section, key, value)
            cfg.write(open(filename, "w"))
        else:
            logger.error('the input key or section is invalid')
            sys.exit(1)

    @staticmethod
    def read_conf_by_file(filename):
        logger.info('read conf by file')
        if not os.path.isfile(filename):
            return ''
        driver_conf = {}
        cfg = config_parser.ConfigParser()
        cfg.read(filename)
        # default
        for key in cfg.defaults().keys():
            v = cfg.get(config_parser.DEFAULTSECT, key)
            if key in [EBACKUP_SERVER_PASSWORD_V2, 'ebackup_storage_username_v2', 'ebackup_storage_password_v2']:
                if six.PY3:
                    v = str(base64.b64decode(v), encoding='utf-8')
                else:
                    v = base64.b64decode(v)
            driver_conf[('default', key)] = v
        # section ebk_XXX
        for section in cfg.sections():
            for key in cfg.options(section):
                if key not in cfg.defaults():
                    v = cfg.get(section, key)
                    if key in [EBACKUP_SERVER_PASSWORD_V2, 'ebackup_storage_username_v2',
                               'ebackup_storage_password_v2']:
                        if six.PY3:
                            v = str(base64.b64decode(v), encoding='utf-8')
                        else:
                            v = base64.b64decode(v)
                    driver_conf[(section, key)] = v
        return driver_conf

    @staticmethod
    def get_pwd_from_file(pwd_file):
        if not os.path.isfile(pwd_file):
            logger.info('The password file with given path is not exist.')
            sys.exit(1)
        key = None
        fp = open(pwd_file, 'r')
        try:
            line = fp.readline().strip('\n')
            line_context = line.split('=', 1)
            if len(line_context) == 2:
                key = line_context[0].strip(' ')
                ori_value = line_context[1].strip(' ')
                if six.PY3:
                    value = str(base64.b64decode(ori_value), encoding='utf-8')
                else:
                    value = base64.b64decode(ori_value)
                return key, value
            else:
                logger.info('the pwd file is informat')
                sys.exit(1)
        finally:
            fp.close()
            if not key == 'cps_userpasswd':
                logger.info("Remove the password file.")
                os.remove(pwd_file)

    @staticmethod
    def _check_ebk_param(filename, cps_name, cps_pwd):
        logger.info('check ebk param')
        if not os.path.isfile(filename):
            logger.info(filename + ' not exist')
            return True
        cfg = config_parser.ConfigParser()
        cfg.read(filename)
        cpslinker = CpsConfig('', '', cps_name, cps_pwd)
        ext_parameters = cpslinker.get_backup_extend_parameters()
        for section in cfg.sections():
            ip = ext_parameters.get(section + '.' + EBACKUP_SERVER_IP, '')
            port = ext_parameters.get(section + '.' + EBACKUP_SERVER_PORT, '')
            name = ext_parameters.get(section + '.' + EBACKUP_SERVER_USERNAME, '')
            cert = ext_parameters.get(section + '.' + EBACKUP_LOGIN_CRT, '')
            from cinder.backup.drivers import ebackupagent
            pwd = ebackupagent.Utils.decrypt_password(
                ext_parameters.get(section + '.' + EBACKUP_SERVER_PASSWORD_V2, ''))
            logger.info('current conf ip:' + ip + ' port:' + port + ' name:' + name + ' cert:' + cert)
            # if upgrade conf has the key,use upgrade conf value
            for key in cfg.options(section):
                if key == EBACKUP_SERVER_IP:
                    ip = cfg.get(section, key)
                if key == EBACKUP_SERVER_PORT:
                    port = cfg.get(section, key)
                if key == EBACKUP_SERVER_USERNAME:
                    name = cfg.get(section, key)
                if key == EBACKUP_LOGIN_CRT:
                    cert = cfg.get(section, key)
                if key == EBACKUP_SERVER_PASSWORD_V2:
                    pwd_ori = cfg.get(section, key)
                    if six.PY3:
                        pwd = str(base64.b64decode(pwd_ori), encoding='utf-8')
                    else:
                        pwd = base64.b64decode(pwd_ori)
            logger.info('new and add curr conf ip:' + ip + ' port:' + port + ' name:' + name + ' cert:' + cert)
            if ip == '' or port == '' or cert == '' or name == '' or pwd == '':
                logger.info('someone was get null in ip, port, cert, name, pwd')
                return False
            if not check_ebackup_pwd(ip, port, name, pwd):
                return False
            ebk_conf = ebackup_server_config()
            ebk_conf.ebackup_server_ip = ip
            ebk_conf.ebackup_server_port = port
            ebk_conf.ebackup_login_username = name
            ebk_conf.ebackup_login_password = pwd
            ebk_conf.ebackup_login_crt = cert
            ebk_client = EbackupConnectionClient(Connection(ebk_conf))
            statist, err, desc = ebk_client.get_backup_statistics()
            if err != 0 or 'UUID' not in statist.keys():
                logger.error('get ebackup %s\'s UUID failed' % ip)
                return False
            else:
                DmkConfig.write_conf_to_file('uuid', statist['UUID'], section, UPGRADE_FILE)
        logger.info('check ebk param success')
        return True

    @staticmethod
    def upgrade_config_start():
        logger.info('interface start, remove ' + UPGRADE_FILE)
        if os.path.isfile(UPGRADE_FILE):
            os.remove(UPGRADE_FILE)
        os.system('mkdir -p {}'.format(UPGRADE_PATH))
        logger.info('remove ' + UPGRADE_FILE + ' success')

    def upgrade_config_check(self):
        logger.info('interface check, check ebk can connect or not')
        if not self._check_ebk_param(UPGRADE_FILE, self.cps_name, self.cps_pwd_value):
            logger.info('check ebk failed')
            sys.exit(1)
        logger.info('check ebk connect success')

    def upgrade_config_update(self, key, value, section):
        logger.info('interface update, write key:%s, section:%s to %s' % (key, section, UPGRADE_FILE))
        if section != 'default':
            section = get_section_name_with_ip(section)
        # write to file
        self.write_conf_to_file(key, value, section, UPGRADE_FILE)
        logger.info('interface update, write key:%s, section:%s to %s success' % (key, section, UPGRADE_FILE))

    def upgrade_config_append(self, key, value, section):
        logger.info('interface append, write key:%s, section:%s to %s' % (key, section, UPGRADE_FILE))
        if section != 'default':
            section = get_section_name_with_ip(section)
        cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
        # backup origin parameters for rollback
        backup_parameters = cpslinker.get_backup_extend_parameters()
        origin_value = backup_parameters.get(section + "." + key, '')
        if origin_value:
            value_list = value.split(',')
            origin_value_list = origin_value.split(',')
            for k, value_list_r in enumerate(value_list):
                if value_list_r not in origin_value_list:
                    origin_value = origin_value + ',' + value_list_r
            value = origin_value
        # write to file
        self.write_conf_to_file(key, value, section, UPGRADE_FILE)
        logger.info(
            'interface append, write key:%s, section:%s to %s success' % (key, section, UPGRADE_FILE))

    def upgrade_config_commit(self, unused_backup_service, extend_volume_driver):
        logger.info('interface commit, try to commit all conf')
        try:
            parameters = dict()
            parameters['default.backup_driver'] = 'cinder.backup.drivers.hwsebackup'
            parameters['default.backup_service_outside'] = 'True'
            parameters['default.ebackup_task_retry_count'] = '90'
            parameters['default.ebackup_task_retry_time_interval'] = '30'
            ebackup_servers = list()
            # update item in upgrade conf
            upgrade_conf = self.read_conf_by_file(UPGRADE_FILE)
            for (section, key) in upgrade_conf:
                if not self.check_param_for_dmk(key, upgrade_conf[(section, key)]):
                    sys.exit(1)
                parameters_key = section + '.' + key
                parameters[parameters_key] = upgrade_conf[(section, key)]
                if section != 'default' and section not in ebackup_servers:
                    ebackup_servers.append(section)
            cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
            # backup origin parameters for rollback
            logger.info('start backup')
            backup_parameters = cpslinker.get_backup_extend_parameters()
            bak_file = open('/home/fsp/ebackup_conf.bak', 'w')
            logger.info('%s' % bak_file)
            json.dump(backup_parameters, bak_file, sort_keys=False, indent=4, separators=(',', ':'))
            bak_file.close()
            if backup_parameters.get('default.ebackup_servers'):
                exist_servers = backup_parameters.get('default.ebackup_servers').split()
                ebackup_servers = list(set(ebackup_servers).union(set(exist_servers)))
            parameters['default.ebackup_servers'] = ",".join(ebackup_servers)
            # write new parameters to cps
            cpslinker.set_backup_extend_parameters(parameters, unused_backup_service.split(),
                                                   extend_volume_driver.split(),
                                                   dispatch_params_to_all=True)
            logger.info('commit all conf success')
            self.sync_config_to_all_template( unused_backup_service, extend_volume_driver)
        finally:
            if os.path.exists(self.cps_file):
                os.remove(self.cps_file)
            if os.path.exists(UPGRADE_FILE):
                os.remove(UPGRADE_FILE)

    def sync_config_to_all_template(self, unused_backup_service, extend_volume_driver):
        logger.info('interface sync, try to sync all template')
        from cinder.backup.drivers import ebackupagent
        try:
            cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
            logger.info('start get config')
            backup_parameters = cpslinker.get_backup_extend_parameters()
            logger.info('Check UUID for each eBackup Server conf')
            ebackup_servers = backup_parameters.get('default.ebackup_servers', '')
            if ebackup_servers:
                server_list = ebackup_servers.split(",")
                for server in server_list:
                    if not backup_parameters.get(server + '.uuid', ''):
                        ebk_conf = ebackup_server_config()
                        ebk_conf.ebackup_server_ip = backup_parameters.get(server + '.' + EBACKUP_SERVER_IP)
                        ebk_conf.ebackup_server_port = backup_parameters.get(server + '.' + EBACKUP_SERVER_PORT)
                        ebk_conf.ebackup_login_username = backup_parameters.get(server + '.' + EBACKUP_SERVER_USERNAME)
                        ebk_conf.ebackup_login_password = ebackupagent.Utils.decrypt_password(
                            backup_parameters.get(server + '.' + EBACKUP_SERVER_PASSWORD_V2, ''))
                        ebk_conf.ebackup_login_crt = backup_parameters.get(server + '.' + EBACKUP_LOGIN_CRT)
                        ebk_client = EbackupConnectionClient(Connection(ebk_conf))
                        statist, err, desc = ebk_client.get_backup_statistics()
                        backup_parameters[server + '.uuid'] = statist.get('UUID', '')
                        logger.info('fill up UUID %s for eBackup %s!' % (statist.get('UUID', ''), server))
                    else:
                        logger.info(
                            'already exist eBackup %s UUID %s!' % (server, backup_parameters.get(server + '.uuid')))
                    backup_parameters[server + '.' + EBACKUP_SERVER_PASSWORD_V2] = ebackupagent.Utils.decrypt_password(
                        backup_parameters.get(server + '.' + EBACKUP_SERVER_PASSWORD_V2, ''))
            if backup_parameters.get('default.ebackup_storage_username_v2', ''):
                backup_parameters['default.ebackup_storage_username_v2'] = ebackupagent.Utils.decrypt_password(
                    backup_parameters.get('default.ebackup_storage_username_v2', ''))
            if backup_parameters.get('default.ebackup_storage_password_v2', ''):
                backup_parameters['default.ebackup_storage_password_v2'] = ebackupagent.Utils.decrypt_password(
                    backup_parameters.get('default.ebackup_storage_password_v2', ''))
            if not backup_parameters.get('default.backup_service_outside', ''):
                backup_parameters['default.backup_service_outside'] = 'True'
            logger.info('start sync config to all template')
            cpslinker.set_backup_extend_parameters(backup_parameters, unused_backup_service.split(),
                                                   extend_volume_driver.split(),
                                                   dispatch_params_to_all=True)
            logger.info('sync all conf success')
        finally:
            if os.path.exists(self.cps_file):
                os.remove(self.cps_file)
            if os.path.exists(UPGRADE_FILE):
                os.remove(UPGRADE_FILE)

    def upgrade_config_rollback(self):
        try:
            cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
            bak_file = open('/home/fsp/ebackup_conf.bak', 'r')
            backup_parameters = json.load(bak_file)
            for k, v in backup_parameters.items():
                if 'ebackup_storage_password_v2' in k or 'ebackup_storage_username_v2' in k or 'ebackup_login_password_v2' in k:
                    try:
                        from cinder.backup.drivers import ebackupagent
                        backup_parameters[k] = ebackupagent.Utils.decrypt_password(v)
                    except Exception:
                        logger.info('warning decrypt failed!')
            cpslinker.set_backup_extend_parameters(backup_parameters, full_cover=True)
        finally:
            if os.path.exists(self.cps_file):
                os.remove(self.cps_file)
            if os.path.exists(UPGRADE_FILE):
                os.remove(UPGRADE_FILE)

    def uninstall_all_parameters(self):
        cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
        backup_parameters = dict()
        backup_parameters['default.backup_driver'] = 'cinder.backup.drivers.swift'
        cpslinker.set_backup_extend_parameters(backup_parameters, full_cover=True)


def get_asterisk_ch():
    fd = sys.stdin.fileno()
    old_setting = termios.tcgetattr(fd)
    try:
        tty.setraw(sys.stdin.fileno())
        ch = sys.stdin.read(1)
    finally:
        termios.tcsetattr(fd, termios.TCSADRAIN, old_setting)
    return ch


def get_asterisk_password(mask="*"):
    password = ""
    while True:
        ch = get_asterisk_ch()
        if ch == "\r" or ch == "\n":
            print('')
            return password
        elif ch == '\b' or ord(ch) == 127:
            if len(password) > 0:
                sys.stdout.write('\b \b')
                password = password[:-1]
        else:
            if mask is not None:
                sys.stdout.write(mask)
            password += ch


def check_valid_char(params):
    special_str = "|;&$><`\\"
    if isinstance(params, int):
        return True
    for ch in special_str:
        if ch in params:
            return False
    return True


class CpsConfig(object):
    """ cps config """

    def __init__(self, authurl, baseurl, cps_user, cps_password):
        self._connect = cpscli.CpsConnection(authurl, baseurl, cps_user, cps_password)

    def set_backup_extend_parameters(self, parameters, unused_template=None,
                                     extend_support_volume_drivers=None,
                                     dispatch_params_to_all=False,
                                     full_cover=False):
        logger.debug('force rollback cache first!')
        self._connect.rollback()
        exist_backup_templates = self._connect.get_cinder_backup_templates()
        available_backup_templates = \
            self._filter_unavailable_template(exist_backup_templates,
                                              unused_template,
                                              extend_support_volume_drivers)
        for template in available_backup_templates:
            logger.debug('start check template %s' % template)
            template_ext_params = \
                self._connect.get_backup_template_ext_params(template)
            if template_ext_params.get('default.backup_driver') == \
                    'cinder.backup.drivers.hwsebackup':
                logger.debug("update hwsebackup template's parameters : %s"
                             % template)
                self._connect.set_backup_template_ext_params(template,
                                                             parameters,
                                                             full_cover)
            elif dispatch_params_to_all is True:
                logger.debug('New configuration parameters to %s' % template)
                self._connect.set_backup_template_ext_params(template,
                                                             parameters,
                                                             full_cover)
            else:
                logger.debug('not dispatch config to template: %s' % template)
        self._connect.commit()

    def _filter_unavailable_template(self, backup_templates,
                                     unused_templates=None,
                                     ext_support_volume_drivers=None):
        # 过滤掉不用的 templates
        unused_templates = unused_templates if unused_templates else []
        logger.debug("unused templates: %s will be filtered out."
                     % unused_templates)
        backup_templates = \
            [t for t in backup_templates if t not in unused_templates]

        # 检查 template_params 或 other_storage_cfg 中的 volume_driver 是否
        # 至少有 1 个在 valid_volume_driver 中，如果都不在，也过滤掉
        ext_support_volume_drivers = \
            ext_support_volume_drivers if ext_support_volume_drivers else []
        valid_volume_driver = \
            VALID_VOLUME_DRIVER + ext_support_volume_drivers
        result = []
        for template in backup_templates:
            logger.debug('start check template %s' % template)
            template_params = \
                self._connect.get_backup_template_params(template)
            volume_driver = template_params.get('volume_driver', '')
            logger.debug('volume_driver: %s' % volume_driver)
            other_volume_driver = \
                self._get_volume_driver_from_other_storage_cfg(template_params)
            if volume_driver in valid_volume_driver or \
                    other_volume_driver in valid_volume_driver:
                result.append(template)
        return result

    @staticmethod
    def _get_volume_driver_from_other_storage_cfg(template_params):
        other_storage_cfg = template_params.get('other_storage_cfg', '')
        logger.debug('other_storage_cfg: %s' % other_storage_cfg)
        other_volume_driver = ''
        if not other_storage_cfg:
            return other_volume_driver
        for pool_list in other_storage_cfg.values():
            for pool_cfg in pool_list.values():
                if pool_cfg.get('volume_driver', ''):
                    other_volume_driver = pool_cfg.get('volume_driver', '')
        logger.debug('other_volume_driver: %s' % other_volume_driver)
        return other_volume_driver

    def get_backup_extend_parameters(self):
        exist_backup_templates = self._connect.get_cinder_backup_templates()
        available_backup_templates = \
            self._filter_unavailable_template(exist_backup_templates)
        final_ext_params = dict()
        for template in available_backup_templates:
            ext_params = self._connect.get_backup_template_ext_params(template)
            if ext_params.get('default.backup_driver') != \
                    'cinder.backup.drivers.hwsebackup':
                logger.debug('template not configed to ebackup! %s' % template)
                continue
            # 合并各个 template_ext_params 中的 default.ebackup_servers
            pre_servers, cur_servers = [], []
            if ext_params.get("default.ebackup_servers"):
                cur_servers = \
                    ext_params.get("default.ebackup_servers").split(",")
            if final_ext_params.get("default.ebackup_servers"):
                pre_servers = \
                    final_ext_params.get("default.ebackup_servers").split(",")
            cur_servers = list(set(pre_servers + cur_servers))
            final_ext_params["default.ebackup_servers"] = ','.join(cur_servers)

            # ebackup_servers 会合并所有值，但其他相同关键字配置项会取最后一个
            final_ext_params = dict(ext_params, **final_ext_params)
        return final_ext_params

    def delete_backup_extend_parameters(self, unlinked=True, uncommited=False):
        logger.debug('now start delete extend parameters of template')
        exist_backup_templates = self._connect.get_cinder_backup_templates()
        for template in exist_backup_templates:
            template_ext_params = self._connect.get_backup_template_ext_params(template)
            if template_ext_params.get('default.backup_driver', '') == 'cinder.backup.drivers.hwsebackup':
                logger.debug('delete extend parameters for template:' + template)
                if not unlinked:
                    logger.debug('preserve driver link to cinder.backup.drivers.hwsebackup')
                    template_ext_params.pop('default.backup_driver')
                self._connect.delete_backup_template_ext_params(template, template_ext_params)
        if not uncommited:
            self._connect.commit()

    def get_supported_backup_templates(self):
        supported_templates = []
        exist_backup_templates = self._connect.get_cinder_backup_templates()
        for template in exist_backup_templates:
            logger.debug('start check template %s' % template)
            template_params = self._connect.get_backup_template_params(template)
            volume_driver = template_params.get('volume_driver', '')
            other_storage_cfg = template_params.get('other_storage_cfg', '')
            logger.debug('volume_driver: %s' % volume_driver)
            logger.debug('other_storage_cfg: %s' % other_storage_cfg)
            valid_volume_driver = VALID_VOLUME_DRIVER
            other_volume_driver = ''
            if other_storage_cfg:
                for poolid, poollist in other_storage_cfg.items():
                    for cfgkey, poolcfg in poollist.items():
                        if poolcfg.get('volume_driver', ''):
                            other_volume_driver = poolcfg.get('volume_driver', '')
            logger.debug('other_volume_driver: %s' % other_volume_driver)
            if volume_driver in valid_volume_driver or (other_volume_driver in valid_volume_driver):
                supported_templates.append(template)
            else:
                logger.debug('not valid volume driver %s' % volume_driver)
        return supported_templates


class MenuItem():
    def __init__(self, submenu, display_text, display_func, action_func, select_key, config_key):
        self.submenu = submenu
        self.display_text = display_text
        self.display_func = display_func
        self.action_func = action_func
        self.select_key = select_key
        self.config_key = config_key
        self.parent_menu = None
        self.grandparent_menu = None


class InteractiveConfig():
    def __init__(self, cps_name, cps_password):
        self.cps_name = cps_name
        self.cps_pwd_value = cps_password
        self.backup_parameters = dict()
        self.need_full_cover = False
        self.need_update = False
        self.unuse_template_names = []
        self.already_set_unuse_template_names = False

    # Action function
    @staticmethod
    def print_and_select_submenu(menuitem):
        for submenu in menuitem.submenu:
            if submenu.display_text == '':
                submenu.display_func(submenu)
            else:
                logger.info(submenu.display_text)

        i = six.moves.input()
        i = i.strip()

        for submenu in menuitem.submenu:
            if i == submenu.select_key:
                return submenu

        logger.info('Invalid input.')
        return menuitem

    @staticmethod
    def retrun_to_grandparent(menuitem):
        return menuitem.grandparent_menu

    @staticmethod
    def set_by_config_key(menuitem):
        pass

    @staticmethod
    def exit_config(menuitem):
        exit(0)

    # Action function end

    # Disp function
    @staticmethod
    def display_noop(menuitem):
        pass

    @staticmethod
    def display_title(menuitem):
        print(menuitem.display_text)

    # Disp function end

    # Business logic Action function
    @staticmethod
    def action_pass(menuitem):
        exit(0)

    def action_add_ebk(self, menuitem):
        ebackup_conf = ebackup_server_config()
        ipaddr, port, ebackup_user, ebackup_pwd = _input_ebackup_info(ebackup_conf)
        server_section_name = get_section_name_with_ip(ipaddr)
        ebackup_servers = self.backup_parameters.get('default.ebackup_servers', '')
        server_list = [] if not ebackup_servers else ebackup_servers.split(",")
        for server in server_list:
            if server == server_section_name:
                msg = "eBackup server \"%s\" already exist!" % ipaddr
                logger.info(msg)
                return menuitem.grandparent_menu
        cert_path = _input_cert_path(ipaddr, port, '')
        azs_info = _input_azs_info('')
        fusion_storage_ip, physical_network_type, fusion_storage_agent_ip = _input_fusion_storage_info('', '', '')
        extend, tenant_write_list = _input_muti_cluster_extend_info()
        logger.info('Adding eBackup server login information. Please wait...')
        logger.info('Adding an authentication certificate of the eBackup server...')
        server_list.append(server_section_name)
        self.backup_parameters['default.backup_driver'] = 'cinder.backup.drivers.hwsebackup'
        self.backup_parameters['default.ebackup_servers'] = ",".join(server_list)
        self.backup_parameters[server_section_name + '.ebackup_server_ip'] = ipaddr
        self.backup_parameters[server_section_name + '.ebackup_server_port'] = port
        self.backup_parameters[server_section_name + '.ebackup_login_username'] = ebackup_user
        self.backup_parameters[server_section_name + '.ebackup_login_password_v2'] = ebackup_pwd
        self.backup_parameters[server_section_name + '.ebackup_login_crt'] = cert_path
        self.backup_parameters[server_section_name + '.azs'] = azs_info
        self.backup_parameters[server_section_name + '.fusion_storage_agent_ip'] = fusion_storage_agent_ip
        self.backup_parameters[server_section_name + '.fusion_storage_ip'] = fusion_storage_ip
        self.backup_parameters[server_section_name + '.physical_network_type'] = physical_network_type
        self.backup_parameters[server_section_name + '.extend'] = extend
        self.backup_parameters[server_section_name + '.tenant_white_list'] = tenant_write_list
        ebk_conf = ebackup_server_config()
        ebk_conf.ebackup_server_ip = ipaddr
        ebk_conf.ebackup_server_port = port
        ebk_conf.ebackup_login_username = ebackup_user
        ebk_conf.ebackup_login_password = ebackup_pwd
        ebk_conf.ebackup_login_crt = cert_path
        ebk_client = EbackupConnectionClient(Connection(ebk_conf))
        statist, err, desc = ebk_client.get_backup_statistics()
        if err != 0 or 'UUID' not in statist.keys():
            logger.error('get ebackup %s\'s UUID failed.[%s]' % (ipaddr, desc))
            return menuitem.grandparent_menu
        else:
            self.backup_parameters[server_section_name + '.uuid'] = statist['UUID']
        logger.info('Succeeded in adding login information and the authentication certificate.')
        self.need_update = True
        return menuitem.grandparent_menu

    def action_update_ebk(self, menuitem):
        while True:
            ebackup_conf = ebackup_server_config()
            cfg_servers_map = self._get_ebackup_server_list()
            server_index = []
            i = 0
            print_str = ''
            for name, server_item in cfg_servers_map.items():
                i = i + 1
                print_str += _print_ebackup_server(i, server_item)
                server_index.append(name)
            _index = ebackupconfig_raw_input('which one do you want to update?\n' + print_str + '0. Exit\n')
            if not _index.isdigit() or int(_index) > i or int(_index) < 0:
                logger.info('Invalid input.')
            elif _index == CHOOSE_EXIT:
                break
            else:
                select_section_name = server_index[int(_index) - 1]
                select_server_object = cfg_servers_map.get(select_section_name)
                ebackup_conf.ebackup_server_ip = select_server_object.get(EBACKUP_SERVER_IP, "")
                ebackup_conf.ebackup_server_port = select_server_object.get(EBACKUP_SERVER_PORT, "")
                ebackup_conf.ebackup_login_username = select_server_object.get(EBACKUP_SERVER_USERNAME, "")
                ipaddr, port, ebackup_user, ebackup_pwd = _input_ebackup_info(ebackup_conf)
                cert_path = _input_cert_path(ipaddr, port, select_server_object.get(EBACKUP_LOGIN_CRT, ""))
                azs_info = _input_azs_info(select_server_object.get(AZS, ""))
                fusion_storage_ip, physical_network_type, fusion_storage_agent_ip = _input_fusion_storage_info(
                    select_server_object.get(FUSION_STORAGE_IP, ""),
                    select_server_object.get(PHYSICAL_NETWORK_TYPE, ""),
                    select_server_object.get(FUSION_STORAGE_AGENT_IP, ""))
                extend, tenant_write_list = _input_muti_cluster_extend_info()
                server_section_name = get_section_name_with_ip(ipaddr)
                logger.info('Updating the eBackup server. Please wait...')
                if select_section_name != server_section_name:
                    self._delete_ebackup_section(select_section_name)
                    self.need_full_cover = True
                    tmp_servers = []
                    for server in cfg_servers_map:
                        if server != select_section_name:
                            tmp_servers.append(server)
                        else:
                            tmp_servers.append(server_section_name)
                    self.backup_parameters['default.ebackup_servers'] = ",".join(tmp_servers)

                logger.info('Updating an authentication certificate of the eBackup server...')
                self.backup_parameters[server_section_name + '.ebackup_server_ip'] = ipaddr
                self.backup_parameters[server_section_name + '.ebackup_server_port'] = port
                self.backup_parameters[server_section_name + '.ebackup_login_username'] = ebackup_user
                self.backup_parameters[server_section_name + '.ebackup_login_password_v2'] = ebackup_pwd
                self.backup_parameters[server_section_name + '.ebackup_login_crt'] = cert_path
                self.backup_parameters[server_section_name + '.azs'] = azs_info
                self.backup_parameters[server_section_name + '.fusion_storage_agent_ip'] = fusion_storage_agent_ip
                self.backup_parameters[server_section_name + '.fusion_storage_ip'] = fusion_storage_ip
                self.backup_parameters[server_section_name + '.physical_network_type'] = physical_network_type
                self.backup_parameters[server_section_name + '.extend'] = extend
                self.backup_parameters[server_section_name + '.tenant_white_list'] = tenant_write_list
                ebk_conf = ebackup_server_config()
                ebk_conf.ebackup_server_ip = ipaddr
                ebk_conf.ebackup_server_port = port
                ebk_conf.ebackup_login_username = ebackup_user
                ebk_conf.ebackup_login_password = ebackup_pwd
                ebk_conf.ebackup_login_crt = cert_path
                ebk_client = EbackupConnectionClient(Connection(ebk_conf))
                statist, err, desc = ebk_client.get_backup_statistics()
                if err != 0 or 'UUID' not in statist.keys():
                    logger.error('get ebackup %s\'s UUID failed.[%s]' % (ipaddr, desc))
                    return menuitem.grandparent_menu
                else:
                    self.backup_parameters[server_section_name + '.uuid'] = statist['UUID']
                logger.info('Succeeded in updating the eBackup server and authentication certificate.')
                self.need_update = True

        return menuitem.grandparent_menu

    def action_delete_ebk(self, menuitem):
        while True:
            cfg_servers_map = self._get_ebackup_server_list()
            server_index = []
            i = 0
            print_str = ''
            for old_section_name, server_item in cfg_servers_map.items():
                i = i + 1
                print_str += _print_ebackup_server(i, server_item)
                server_index.append(old_section_name)
            _index = ebackupconfig_raw_input('which one do you want to delete?\n' + print_str + '0. Exit\n')
            if not _index.isdigit() or int(_index) > i or int(_index) < 0:
                logger.info('Invalid input.')
            elif _index == CHOOSE_EXIT:
                break
            else:
                select_section_name = server_index[int(_index) - 1]
                confirm = ebackupconfig_raw_input('Are you sure you want to delete this server?[Y/N]')
                if confirm.lower().strip() == 'y' or confirm.lower().strip() == 'yes':
                    logger.info('Deleting the eBackup server. Please wait...')
                    tmp_servers = []
                    for server in cfg_servers_map:
                        if server != select_section_name:
                            tmp_servers.append(server)
                    self.backup_parameters['default.ebackup_servers'] = ",".join(tmp_servers)
                    self._delete_ebackup_section(select_section_name)
                    logger.info('Succeeded in deleting the eBackup server.')
                    self.need_update = True
                    self.need_full_cover = True

        return menuitem.grandparent_menu

    def action_config_ebackup_server_info_hybird_cloud(self, menuitem):
        ebackup_conf = ebackup_server_config()
        cfg_servers_map = self._get_ebackup_server_list()
        old_section_name = ''
        if len(cfg_servers_map) > 1:
            logger.info('Too many eBackup servers are detected in hybrid cloud scenarios. Please perform a check')
            return menuitem.grandparent_menu
        elif len(cfg_servers_map) == 1:
            for old_section_name, value in cfg_servers_map.items():
                ebackup_conf.ebackup_server_ip = value.get(EBACKUP_SERVER_IP, "")
                ebackup_conf.ebackup_server_port = value.get(EBACKUP_SERVER_PORT, "")
                ebackup_conf.ebackup_login_username = value.get(EBACKUP_SERVER_USERNAME, "")
                ebackup_conf.ebackup_login_password = value.get(EBACKUP_SERVER_PASSWORD_V2, "")
                ebackup_conf.ebackup_login_crt = value.get(EBACKUP_LOGIN_CRT, "")
        ipaddr, port, ebackup_user, ebackup_pwd = _input_ebackup_info(ebackup_conf)
        cert_path = _input_cert_path(ipaddr, port, ebackup_conf.ebackup_login_crt)
        server_section_name = get_section_name_with_ip(ipaddr)
        logger.info('Updating the eBackup server. Please wait...')
        logger.info('Updating an authentication certificate of the eBackup server...')
        self.backup_parameters[server_section_name + '.ebackup_server_ip'] = ipaddr
        self.backup_parameters[server_section_name + '.ebackup_server_port'] = port
        self.backup_parameters[server_section_name + '.ebackup_login_username'] = ebackup_user
        self.backup_parameters[server_section_name + '.ebackup_login_password_v2'] = ebackup_pwd
        self.backup_parameters[server_section_name + '.ebackup_login_crt'] = cert_path
        self.backup_parameters['default.ebackup_servers'] = server_section_name
        logger.info('Succeeded in updating the eBackup server and authentication certificate.')
        return menuitem.grandparent_menu

    def action_config_public_env_info(self, menuitem):
        uds_proto, be_proto_modify = self._input_s3_proto()
        uds_addr, be_addr_modify = self._input_uds_server()
        uds_container_name, be_name_modify = self._input_bucket()
        ak, sk = _input_ak_sk()
        uds_data_layout = _input_uds_data_layout()

        be_evs_setting = self._input_be_EVSSetting()
        if be_evs_setting:
            api_gateway_host_name, be_host_modify = self._input_api_gateway_host_name()
            api_gateway_ca_file, be_ca_modify = self._input_api_gateway_ca_file()
            self.backup_parameters['default.api_gateway_host'] = api_gateway_host_name
            self.backup_parameters['default.api_gateway_ca_file'] = api_gateway_ca_file

        be_modify_lazyloading, is_lazyloading = self._input_lazyloading()
        self.backup_parameters['default.ebackup_lazyloading'] = is_lazyloading

        if self._set_CSBS_Tag():
            be_modify_nova_endpoint, nova_endpoint = self._input_nova_endpoint()
            be_modify_ecs_api, nova_ecs_api = self._input_nova_ecs_api()
            self.backup_parameters['default.ebackup_nova_endpoint'] = nova_endpoint
            self.backup_parameters['default.ebackup_nova_ecs_api'] = nova_ecs_api

        self.backup_parameters['default.ebackup_uds_data_layout'] = uds_data_layout
        self.backup_parameters['default.ebackup_storage_unit_path'] = uds_addr
        self.backup_parameters['default.ebackup_bucket_name'] = uds_container_name
        self.backup_parameters['default.ebackup_storage_username_v2'] = ak
        self.backup_parameters['default.ebackup_storage_password_v2'] = sk
        self.backup_parameters['default.ebackup_storage_s3_protocol'] = uds_proto
        self.backup_parameters['default.ebackup_env_type'] = ENV_PUBLIC_CLOUD
        self.backup_parameters['default.ebackup_task_retry_count'] = TASK_RETRY_COUNT
        self.backup_parameters['default.ebackup_task_retry_time_interval'] = TASK_RETRY_TIME_INTERVAL
        self.backup_parameters['default.backup_service_outside'] = 'True'
        self.need_update = True
        logger.info('Configuration succeeded.')

        return menuitem.grandparent_menu

    def action_config_private_env_info(self, menuitem):
        unit_type = UNIT_NAS
        while True:
            unit_type = six.moves.input('Please config the backup storage:\n1: S3\n2: NAS\n0: Exit\n')
            if unit_type == UNIT_UDS:
                uds_data_layout = _input_uds_data_layout()
                self.backup_parameters['default.backup_service_outside'] = 'True'
                ret = self._update_all_copy_policy()
                if ret != 0:
                    return menuitem.grandparent_menu
                self.backup_parameters['default.ebackup_uds_data_layout'] = uds_data_layout
                self.backup_parameters['default.ebackup_env_type'] = ENV_PRIVATE_CLOUD
                self.backup_parameters['default.ebackup_task_retry_count'] = TASK_RETRY_COUNT
                self.backup_parameters['default.ebackup_task_retry_time_interval'] = TASK_RETRY_TIME_INTERVAL
                self.need_update = True
                logger.info('Configuration succeeded.')
            elif unit_type == UNIT_NAS:
                nfs_data_layout, cifs_data_layout = _input_nas_data_layout()
                self.backup_parameters['default.backup_service_outside'] = 'True'
                ret = self._update_all_copy_policy()
                if ret != 0:
                    return menuitem.grandparent_menu
                self.backup_parameters['default.ebackup_nfs_data_layout'] = nfs_data_layout
                self.backup_parameters['default.ebackup_cifs_data_layout'] = cifs_data_layout
                self.backup_parameters['default.ebackup_env_type'] = ENV_PRIVATE_CLOUD
                self.backup_parameters['default.ebackup_task_retry_count'] = TASK_RETRY_COUNT
                self.backup_parameters['default.ebackup_task_retry_time_interval'] = TASK_RETRY_TIME_INTERVAL
                self.need_update = True
                logger.info('Configuration succeeded.')
            elif unit_type == CHOOSE_EXIT:
                ret = self._update_all_copy_policy()
                if ret != 0:
                    return menuitem.grandparent_menu
                break
            else:
                logger.info('Invalid input.')
        return menuitem.grandparent_menu

    def action_exit_and_save(self, menuitem):
        logger.info('quit program.')
        logger.info('configuration eBackup driver finish.')
        if not self.need_update:
            exit(0)

        cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
        cpslinker.set_backup_extend_parameters(self.backup_parameters, unused_template=self.unuse_template_names,
                                               dispatch_params_to_all=True, full_cover=self.need_full_cover)
        exit(0)

    def action_print_config_info(self, menuitem):
        ebackup_server_list = self._get_ebackup_server_list()

        logger.info("======================== Current Configuration ==========================")
        self._print_conf_option('default.ebackup_server_ip', '# eBackup Server Management IP Address :')
        self._print_conf_option('default.ebackup_server_port', '# eBackup Sdefault.erver Port :')
        self._print_conf_option('default.ebackup_login_username', '# eBackup Login Username  :')
        for old_section_name, server_item in ebackup_server_list.items():
            print_ebackup_server_config(server_item)

        self._print_conf_option('default.ebackup_env_type', '# eBackup Environment Type     :')
        env_type = self.backup_parameters.get('default.ebackup_env_type')
        if env_type != '':
            if env_type == ENV_PUBLIC_CLOUD:
                self._print_conf_option('default.ebackup_storage_unit_path',
                                        '# S3 IP Address Or Domain Name  :')
                self._print_conf_option('default.ebackup_bucket_name', '# S3 Bucket Name Prefix  :')
                self._print_conf_option('default.ebackup_uds_data_layout', '# Data Layout Of S3 Buckets  :')
                self._print_conf_option('default.api_gateway_host', '# API-gateway Access URL  :')
                self._print_conf_option('default.api_gateway_ca_file',
                                        '# API-gateway Root Certificate File  :')
                self._print_conf_option('default.ebackup_lazyloading', '# eBackup Lazyloading    :')
                self._print_conf_option('default.ebackup_nova_endpoint', '# eBackup nova endpoint    :')
                self._print_conf_option('default.ebackup_nova_ecs_api', '# eBackup nova ecs api    :')
                # change index to protocol name'
                self._print_s3_protocol()
            elif env_type == ENV_PRIVATE_CLOUD:
                self._print_conf_option('default.ebackup_uds_data_layout',
                                        '# Data Layout Of S3 Storage Units  :')
                self._print_conf_option('default.ebackup_nfs_data_layout',
                                        '# Data Layout Of NFS Storage Units  :')
                self._print_conf_option('default.ebackup_cifs_data_layout',
                                        '# Data Layout Of CIFS Storage Units  :')
                self._print_conf_option('default.ebackup_storage_selection_policy',
                                        '# Policy Of Select Backup Storage  :')
            elif env_type == ENV_HYBRID_CLOUD:
                self._print_conf_option('default.ebackup_image_limit', '# Image Limit  :')
                self._print_conf_option('default.ebackup_copy_time', '# Copy Time  :')
                self._print_conf_option('default.ebackup_copy_day_in_week', '# Copy Day In Week  :')
                self._print_conf_option('default.ebackup_uds_data_layout', '# Data Layout Of S3 Buckets  :')
                self._print_conf_option('default.ebackup_nfs_data_layout',
                                        '# Data Layout Of NFS Storage Units  :')
                self._print_conf_option('default.ebackup_cifs_data_layout',
                                        '# Data Layout Of CIFS Storage Units  :')
                self._print_conf_option('default.ebackup_san_data_layout',
                                        '# Data Layout Of SAN Storage Units  :')
        logger.info('======================== Current Configuration ==========================')

        return menuitem.grandparent_menu

    def action_set_could_backup_template_names(self, menuitem):
        return self.print_and_select_submenu(menuitem)

    def action_config_ebackup_server_info_cloud(self, menuitem):
        self._move_ebackup_server_config_to_outside()
        return self.print_and_select_submenu(menuitem)

    # Business logic Action function end

    def _input_s3_proto(self):
        be_modify = True
        while True:
            input_req = 'Please enter the protocol of the S3 service plane(0:https,1:http):'
            is_exist = 'ebackup_storage_s3_protocol' in self.backup_parameters
            if not is_exist:
                be_modify = False
                default_proto = '0'
            else:
                default_proto = self.backup_parameters.get('ebackup_storage_s3_protocol', '')
            _s3_proto = six.moves.input(input_req + "(default:" + default_proto + ")\n")
            if '' == _s3_proto.strip():
                be_modify = False
                _s3_proto = default_proto
            if _s3_proto not in ['0', '1']:
                logger.info('Invalid input. Please check it.')
                continue

            if _s3_proto == '0':
                logger.info(
                    'You are advised to import the S3 certificate to the eBackup software, which is more secure.\n')
            if _s3_proto == '1':
                choose = six.moves.input(
                    "Security risks arise if the protocol is set to HTTP. You are advised to select the HTTPS protocol, are you sure to continue?[Y/N](default:n):")
                if choose not in ['Y', 'y']:
                    continue
            break
        return _s3_proto, be_modify

    def _input_uds_server(self):
        be_modify = True
        while True:
            input_req = 'Please enter the domain name or IP address of the S3 service plane:'
            is_exist = 'ebackup_storage_unit_path' in self.backup_parameters
            if not is_exist:
                be_modify = False
                default_addr = 'None'
            else:
                default_addr = self.backup_parameters.get('ebackup_storage_unit_path', '')
            _uds_addr = six.moves.input(input_req + "(default:" + default_addr + ")\n")
            if _uds_addr.strip() == '':
                if default_addr == 'None':
                    logger.info('Can not be blank.')
                    continue
                _uds_addr = default_addr
                be_modify = False
            break
        return _uds_addr, be_modify

    def _input_bucket(self):
        be_modify = True
        while True:
            input_req = 'Please enter the prefix for the S3 bucket name:'
            is_exist = 'ebackup_bucket_name' in self.backup_parameters
            if not is_exist:
                be_modify = False
                default_uds_name = 'vbs'
            else:
                default_uds_name = self.backup_parameters.get('ebackup_bucket_name', '')
            _uds_container_name = six.moves.input(input_req + "(default:" + default_uds_name + ")\n")
            if _uds_container_name.strip() == '':
                if default_uds_name == '':
                    logger.info('Can not be blank.')
                    continue
                _uds_container_name = default_uds_name
                be_modify = False
            break
        return _uds_container_name, be_modify

    def _input_api_gateway_host_name(self):
        while True:
            be_modify = True
            input_req = 'Enter the URL for accessing the API-gateway:'
            is_exist = 'api_gateway_host' in self.backup_parameters
            if not is_exist:
                be_modify = True
                default_host_name = 'https://{IP}:{Port} or https://{DomainName}:{Port}'
            else:
                default_host_name = self.backup_parameters.get('api_gateway_host', '')
            api_gateway_host_name = six.moves.input(input_req + "(default:" + default_host_name + ")\n")
            if api_gateway_host_name.strip() == '':
                if default_host_name == 'https://{IP}:{Port} or https://{DomainName}:{Port}':
                    logger.info("Can not be blank.")
                    continue
                api_gateway_host_name = default_host_name
                be_modify = False
            if not api_gateway_host_name.startswith('https://') or len(api_gateway_host_name) > 1024:
                logger.info("The entered information is incorrect or does not meet requirements. Please retry.")
                continue
            return api_gateway_host_name, be_modify

    def _input_api_gateway_ca_file(self):
        while True:
            input_req = 'Enter the root certificate of the API-gateway:'
            be_modify = True
            is_exist = 'api_gateway_ca_file' in self.backup_parameters
            if not is_exist:
                default_api_gateway_ca_file = '/usr/lib/python2.7/site-packages/cinder/backup/ca.crt'
            else:
                default_api_gateway_ca_file = self.backup_parameters.get('api_gateway_ca_file', '')
            attention = ("(default:" + default_api_gateway_ca_file + ")")
            api_gateway_ca_file = six.moves.input(input_req + attention + '\n')
            if api_gateway_ca_file.strip() == '' and is_exist:
                be_modify = False
                api_gateway_ca_file = default_api_gateway_ca_file
            elif api_gateway_ca_file.strip() == '' and not is_exist:
                be_modify = True
                api_gateway_ca_file = default_api_gateway_ca_file
            if not api_gateway_ca_file.startswith('/'):
                logger.info('The entered path must be an absolute path. Please check.')
                continue
            return api_gateway_ca_file, be_modify

    def _input_lazyloading(self):
        is_exist = 'ebackup_lazyloading' in self.backup_parameters
        while True:
            be_modify = True
            if not is_exist:
                default_lazyloading = '0'
            else:
                default_lazyloading = self.backup_parameters.get('ebackup_lazyloading', '')

            is_lazyloading = six.moves.input(
                "Do you want to enable the lazyloading function?(0: Not "
                "supported. 1: supported. 2: supported(ignore op_gated_bklld) "
                "3: supported(include native image) 4: supported(include "
                "native image and ignore op_gated_bklld). "
                "default:" + default_lazyloading + "):\n")
            if is_lazyloading.strip() == '':
                is_lazyloading = default_lazyloading
                if is_lazyloading.strip() in ['1', '2', '3', '4']:
                    logger.info(
                        'Note: After the function is enabled, you need to configure fusionstorage agent ip to make lazyloading take effect. ')
                if is_exist:
                    be_modify = False
                    break
                else:
                    be_modify = True
                    break
            elif is_lazyloading.strip() in ['1', '2', '3', '4']:
                logger.info(
                    'Note: After the function is enabled, you need to configure fusionstorage agent ip to make lazyloading take effect. ')
                break
            elif is_lazyloading.strip() == '0':
                break
            else:
                logger.info('The entered information is incorrect or does not meet requirements. Please retry.')
        return be_modify, is_lazyloading

    def _input_nova_endpoint(self):
        while True:
            be_modify = True
            input_req = 'Enter the URL for accessing the csd nova endpoint:'
            is_exist = 'ebackup_nova_endpoint' in self.backup_parameters
            if not is_exist:
                be_modify = True
                default_nova_endpoint = 'https://{DomainName}:{Port}/{Version}'
            else:
                default_nova_endpoint = self.backup_parameters.get('ebackup_nova_endpoint', '')
            nova_endpoint = six.moves.input(input_req + "(default:" + default_nova_endpoint + ")\n")
            if nova_endpoint.strip() == '':
                if default_nova_endpoint == 'https://{DomainName}:{Port}/{Version}':
                    logger.info("Can not be blank.")
                    continue
                nova_endpoint = default_nova_endpoint
                be_modify = False
            if not nova_endpoint.startswith('https://') or len(nova_endpoint) > 1024:
                logger.info("The entered information is incorrect or does not meet requirements. Please retry.")
                continue
            return be_modify, nova_endpoint

    def _input_nova_ecs_api(self):
        while True:
            be_modify = True
            input_req = 'Enter the URL for accessing the csd nova ecs api:'
            is_exist = 'ebackup_nova_ecs_api' in self.backup_parameters
            if not is_exist:
                be_modify = True
                default_nova_ecs_api = 'https://{DomainName}:{Port}/{Version}'
            else:
                default_nova_ecs_api = self.backup_parameters.get('ebackup_nova_ecs_api', '')
            nova_ecs_api = six.moves.input(input_req + "(default:" + default_nova_ecs_api + ")\n")
            if nova_ecs_api.strip() == '':
                if default_nova_ecs_api == 'https://{DomainName}:{Port}/{Version}':
                    logger.info("Can not be blank.")
                    continue
                nova_ecs_api = default_nova_ecs_api
                be_modify = False
            if not nova_ecs_api.startswith('https://') or len(nova_ecs_api) > 1024:
                logger.info("The entered information is incorrect or does not meet requirements. Please retry.")
                continue
            return be_modify, nova_ecs_api

    def _set_CSBS_Tag(self):
        while True:
            beNeedEdit = six.moves.input('Whether needs to set CSBS Tag?(Y/N default:N):\n')
            if beNeedEdit.strip() == 'N' or beNeedEdit.strip() == 'n' or beNeedEdit.strip() == '':
                return False
            elif beNeedEdit.strip() == 'Y' or beNeedEdit.strip() == 'y':
                return True
            else:
                logger.info('The entered information is incorrect or does not meet requirements. Please retry.')

    def _input_be_EVSSetting(self):
        while True:
            beNeedEdit = six.moves.input('Whether backup data needs to be encrypted?(Y/N default:N):\n')
            if beNeedEdit.strip() == 'N' or beNeedEdit.strip() == 'n' or beNeedEdit.strip() == '':
                return False
            elif beNeedEdit.strip() == 'Y' or beNeedEdit.strip() == 'y':
                return True
            else:
                logger.info('The entered information is incorrect or does not meet requirements. Please retry.')

    def _update_all_copy_policy(self):
        from cinder.backup.drivers import hwsebackup as ebackup
        from cinder.backup.drivers import ebackupagent
        ebackupagent.g_in_ebackupconfig = True
        ebackup_server_ip = self.backup_parameters.get('default.ebackup_server_ip', '')
        if ebackup_server_ip == '':
            ebackup_conf_mutil = self._get_ebackup_server_list()
            if {} == ebackup_conf_mutil:
                logger.info("update copy policy failed, please configure the eBackup server information firstly.")
                return -1
            else:
                for old_section_name, ebackupServer in ebackup_conf_mutil.items():
                    ebackup_conf = ebackup_server_config()
                    ebackup_conf.ebackup_server_ip = ebackupServer[EBACKUP_SERVER_IP]
                    ebackup_conf.ebackup_server_port = ebackupServer[EBACKUP_SERVER_PORT]
                    ebackup_conf.ebackup_login_username = ebackupServer[EBACKUP_SERVER_USERNAME]
                    ebackup_conf.ebackup_login_password = ebackupServer[EBACKUP_SERVER_PASSWORD_V2]
                    ebackup_conf.ebackup_login_crt = ebackupServer[EBACKUP_LOGIN_CRT]
                    ebackup_conn = ebackupagent.Connection(ebackup_conf)
                    error_code, msg = ebackup.EbackupDriver.update_all_copy_policy(ebackup_conn)
                    if 0 != error_code:
                        error_msg = json.loads(msg)
                        error_desc = "Create copy policy failed"
                        if "desc" in error_msg:
                            error_desc = error_msg["desc"]
                        if "code" in error_msg:
                            error_desc = error_desc + ", error number is: " + error_msg["code"]
                        logger.info(error_desc)
                        return error_code
                return 0
        else:
            ebackup_conf = ebackup_server_config()
            ebackup_conf.ebackup_server_ip = self.backup_parameters.get('default.ebackup_server_ip', '')
            ebackup_conf.ebackup_login_crt = self.backup_parameters.get('default.ebackup_login_crt', '')
            ebackup_conf.ebackup_server_port = self.backup_parameters.get('default.ebackup_server_port', '')
            ebackup_conf.ebackup_login_username = self.backup_parameters.get('default.ebackup_login_username', '')
            ebackup_conf.ebackup_login_password = self.backup_parameters.get('default.ebackup_login_password_v2', '')

            ebackup_conn = ebackupagent.Connection(ebackup_conf)
            error_code, msg = ebackup.EbackupDriver.update_all_copy_policy(ebackup_conn)
            if 0 != error_code:
                error_msg = json.loads(msg)
                error_desc = "Create copy policy failed"
                if "desc" in error_msg:
                    error_desc = error_msg["desc"]
                if "code" in error_msg:
                    error_desc = error_desc + ", error number is: " + error_msg["code"]
                logger.info(error_desc)
            return error_code

    def _print_s3_protocol(self):
        s3_protocol_name = "HTTPs"
        if 'ebackup_storage_s3_protocol' in self.backup_parameters:
            s3_protocol_value = self.backup_parameters.get['ebackup_storage_s3_protocol', '']
            if s3_protocol_value == '1':  # http
                s3_protocol_name = 'HTTP'

        print_msg = '# Protocol Of S3 Buckets  :' + str(s3_protocol_name)  # need show protocol name
        logger.info(print_msg)

    def _print_conf_option(self, option, desc):
        if option in self.backup_parameters:
            option_value = self.backup_parameters.get(option, '')
            display_str = desc + str(option_value)
            logger.info(display_str)

    def _get_ebackup_server_list(self):
        backup_parameters = self.backup_parameters
        cfg_servers_map = {}
        ebackup_servers = backup_parameters.get('default.ebackup_servers', '')
        server_list = ebackup_servers.split(",")
        for server in server_list:
            cfg_server = {}
            for key in backup_parameters:
                if key == (server + '.' + EBACKUP_SERVER_IP):
                    cfg_server[EBACKUP_SERVER_IP] = backup_parameters.get(key, '')
                elif key == (server + '.' + EBACKUP_SERVER_PORT):
                    cfg_server[EBACKUP_SERVER_PORT] = backup_parameters.get(key, '')
                elif key == (server + '.' + EBACKUP_SERVER_USERNAME):
                    cfg_server[EBACKUP_SERVER_USERNAME] = backup_parameters.get(key, '')
                elif key == (server + '.' + EBACKUP_SERVER_PASSWORD_V2):
                    cfg_server[EBACKUP_SERVER_PASSWORD_V2] = backup_parameters.get(key, '')
                elif key == (server + '.' + EBACKUP_LOGIN_CRT):
                    cfg_server[EBACKUP_LOGIN_CRT] = backup_parameters.get(key, '')
                elif key == (server + '.' + AZS):
                    cfg_server[AZS] = backup_parameters.get(key, '')
                elif key == (server + '.' + FUSION_STORAGE_IP):
                    cfg_server[FUSION_STORAGE_IP] = backup_parameters.get(key, '')
                elif key == (server + '.' + FUSION_STORAGE_AGENT_IP):
                    cfg_server[FUSION_STORAGE_AGENT_IP] = backup_parameters.get(key, '')
                elif key == (server + '.' + PHYSICAL_NETWORK_TYPE):
                    cfg_server[PHYSICAL_NETWORK_TYPE] = backup_parameters.get(key, '')
            cfg_servers_map[server] = cfg_server
        return cfg_servers_map

    def _delete_ebackup_section(self, server_section_name):
        new_backup_parameters = {}
        for key in self.backup_parameters:
            split_key = key.split('.')
            if not (len(split_key) == 2 and split_key[0] == server_section_name):
                new_backup_parameters[key] = self.backup_parameters[key]
        self.backup_parameters = new_backup_parameters

    def _move_ebackup_server_config_to_outside(self):
        if not self.backup_parameters.get('default.ebackup_server_ip', ''):
            logger.debug('ebackup_server_ip is empty, no old cfg found')
            return
        from cinder.backup.drivers import ebackupagent
        logger.info('Updating configurations. Please wait...')

        ebk_pwd = self.backup_parameters.get('default.ebackup_login_password_v2', '')

        ebackup_server_ip = self.backup_parameters.get('default.ebackup_server_ip', '')
        ebackup_login_crt = self.backup_parameters.get('default.ebackup_login_crt', '')
        ebackup_server_port = self.backup_parameters.get('default.ebackup_server_port', '')
        ebackup_login_username = self.backup_parameters.get('default.ebackup_login_username', '')
        ebackup_login_password = ebackupagent.Utils.decrypt_password(ebk_pwd)
        server_section_name = get_section_name_with_ip(ebackup_server_ip)

        cfg_servers_map = self._get_ebackup_server_list()
        server_list = []
        for name, server_item in cfg_servers_map.items():
            if name != server_section_name:
                server_list.append(name)

        server_list.append(server_section_name)
        self.backup_parameters['default.ebackup_servers'] = ",".join(server_list)
        self.backup_parameters[server_section_name + '.ebackup_server_ip'] = ebackup_server_ip
        self.backup_parameters[server_section_name + '.ebackup_server_port'] = ebackup_server_port
        self.backup_parameters[server_section_name + '.ebackup_login_username'] = ebackup_login_username
        self.backup_parameters[server_section_name + '.ebackup_login_password_v2'] = ebackup_login_password
        self.backup_parameters[server_section_name + '.ebackup_login_crt'] = ebackup_login_crt
        self.backup_parameters[server_section_name + '.azs'] = 'default'
        self.backup_parameters[server_section_name + '.fusion_storage_agent_ip'] = ''
        self.backup_parameters[server_section_name + '.fusion_storage_ip'] = ''
        self.backup_parameters[server_section_name + '.physical_network_type'] = ''

        default_cfg_names = ['default.ebackup_server_ip', 'default.ebackup_server_port', 'default.ebackup_login_crt',
                             'default.ebackup_login_username', 'default.ebackup_login_password_v2']
        for item_name in default_cfg_names:
            if item_name in self.backup_parameters:
                self.backup_parameters.pop(item_name)
        self.need_update = True
        self.need_full_cover = True
        logger.info('Configurations updated.')

    def init_and_get_parameters(self):
        from cinder.backup.drivers import ebackupagent
        cpslinker = CpsConfig('', '', self.cps_name, self.cps_pwd_value)
        self.backup_parameters = cpslinker.get_backup_extend_parameters()
        for key, value in self.backup_parameters.items():
            if key.find('password') != -1 or key.find('storage_username') != -1:
                pwd = ebackupagent.Utils.decrypt_password(value)
                self.backup_parameters[key] = pwd

    def begin_config(self):
        root = MenuItem([],
                        'which one do you want to select?\n(you must configure the eBackup server information for the first time)',
                        InteractiveConfig.display_title, InteractiveConfig.print_and_select_submenu, "", "")

        # View
        view_current = MenuItem([], '1: View Current Configuration',
                                InteractiveConfig.display_noop, self.action_print_config_info, '1', '')
        view_current.grandparent_menu = root

        # Config ebackup
        cfg_ebK = MenuItem([], '2: Configure eBackup Server Information',
                           InteractiveConfig.display_noop, self.action_set_could_backup_template_names, '2', '')
        cfg_ebk_pub = MenuItem([],
                               'which one do you want to select?\n1: Configure eBackup Server Information in Public Cloud Scenarios',
                               InteractiveConfig.display_noop, self.action_config_ebackup_server_info_cloud, '1', '')

        cfg_ebk_pub_add = MenuItem([],
                                   'which one do you want to select?\n1: Add eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_add_ebk, '1', '')
        cfg_ebk_pub_mod = MenuItem([],
                                   '2: Modify eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_update_ebk, '2', '')
        cfg_ebk_pub_del = MenuItem([],
                                   '3: Delete eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_delete_ebk, '3', '')
        cfg_ebk_pub_exit = MenuItem([],
                                    '0: Exit',
                                    InteractiveConfig.display_noop, InteractiveConfig.retrun_to_grandparent, '0', '')
        cfg_ebk_pub_add.grandparent_menu = cfg_ebk_pub
        cfg_ebk_pub_mod.grandparent_menu = cfg_ebk_pub
        cfg_ebk_pub_del.grandparent_menu = cfg_ebk_pub
        cfg_ebk_pub_exit.grandparent_menu = cfg_ebK
        cfg_ebk_pub.submenu = [cfg_ebk_pub_add, cfg_ebk_pub_mod, cfg_ebk_pub_del, cfg_ebk_pub_exit]

        cfg_ebk_pri = MenuItem([], '2: Configure eBackup Server Information in Private Cloud Scenarios',
                               InteractiveConfig.display_noop,
                               self.action_config_ebackup_server_info_cloud, '2', '')

        cfg_ebk_pri_add = MenuItem([],
                                   'which one do you want to select?\n1: Add eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_add_ebk, '1', '')
        cfg_ebk_pri_mod = MenuItem([],
                                   '2: Modify eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_update_ebk, '2', '')
        cfg_ebk_pri_del = MenuItem([],
                                   '3: Delete eBackup Server Configuration Information',
                                   InteractiveConfig.display_noop, self.action_delete_ebk, '3', '')
        cfg_ebk_pri_exit = MenuItem([],
                                    '0: Exit',
                                    InteractiveConfig.display_noop, InteractiveConfig.retrun_to_grandparent, '0', '')

        cfg_ebk_pri_add.grandparent_menu = cfg_ebk_pri
        cfg_ebk_pri_mod.grandparent_menu = cfg_ebk_pri
        cfg_ebk_pri_del.grandparent_menu = cfg_ebk_pri
        cfg_ebk_pri_exit.grandparent_menu = cfg_ebK

        cfg_ebk_pri.submenu = [cfg_ebk_pri_add, cfg_ebk_pri_mod, cfg_ebk_pri_del, cfg_ebk_pri_exit]

        cfg_ebk_hyb = MenuItem([],
                               '3: Configure eBackup Server Information in Hybrid Cloud Scenarios',
                               InteractiveConfig.display_noop, self.action_config_ebackup_server_info_hybird_cloud, '3',
                               '')
        cfg_ebk_exit = MenuItem([],
                                '0: Exit',
                                InteractiveConfig.display_noop, InteractiveConfig.retrun_to_grandparent, '0', '')
        cfg_ebk_exit.grandparent_menu = root
        cfg_ebk_hyb.grandparent_menu = cfg_ebK

        cfg_ebK.submenu = [cfg_ebk_pub, cfg_ebk_pri, cfg_ebk_hyb, cfg_ebk_exit]

        # Config env
        cfg_env = MenuItem([],
                           '3: Configure Backup Environment Information',
                           InteractiveConfig.display_noop, self.action_set_could_backup_template_names, '3', '')
        cfg_env_pub = MenuItem([],
                               'which one do you want to select?\n1: Configure Public Cloud Information',
                               InteractiveConfig.display_noop, self.action_config_public_env_info, '1', '')

        cfg_env_pri = MenuItem([],
                               '2: Configure Private Cloud Information',
                               InteractiveConfig.display_noop, self.action_config_private_env_info, '2', '')
        cfg_env_hyb = MenuItem([],
                               '3: Configure Hybrid Cloud Information',
                               InteractiveConfig.display_noop, self.action_config_private_env_info, '3', '')
        cfg_env_exit = MenuItem([],
                                '0: Exit',
                                InteractiveConfig.display_noop, InteractiveConfig.retrun_to_grandparent, '0', '')

        cfg_env_pub.grandparent_menu = cfg_env
        cfg_env_pri.grandparent_menu = cfg_env
        cfg_env_hyb.grandparent_menu = cfg_env
        cfg_env_exit.grandparent_menu = root
        cfg_env.submenu = [cfg_env_pub, cfg_env_pri, cfg_env_hyb, cfg_env_exit]

        # Exit
        exit_cfg = MenuItem([],
                            '0: Exit',
                            InteractiveConfig.display_noop, self.action_exit_and_save, '0', '')

        root.submenu = [view_current, cfg_ebK, cfg_env, exit_cfg]

        current_menu = root
        while True:
            current_menu.display_func(current_menu)
            current_menu = current_menu.action_func(current_menu)


def decode_from_base64(string_encoded):
    if six.PY3:
        return str(base64.b64decode(string_encoded).decode())
    else:
        return base64.b64decode(string_encoded)


if __name__ == '__main__':
    if sys.argv.__len__() == 2 and os.path.isfile(sys.argv[1]):
        filecfg = FileConfig(sys.argv[1])
        filecfg.file_config_commit()

    # interface dmk start, remove driver_upgrade.conf
    elif sys.argv.__len__() >= 2 and sys.argv[1] == "start":
        DmkConfig.upgrade_config_start()

    # interface update, write key, value, section to driver_upgrade.conf
    elif sys.argv.__len__() >= 2 and sys.argv[1] == "update":
        if sys.argv.__len__() == 5 and \
                sys.argv[3] in ["ebackup_login_password_v2",
                                "ebackup_storage_username_v2",
                                "ebackup_storage_password_v2"]:
            sensitive_info = six.moves.input()
            first_sensitive, second_sensitive = sensitive_info.split(" ")
            cps_password = decode_from_base64(first_sensitive)
            second_sensitive = decode_from_base64(second_sensitive)
            dmkcfg = DmkConfig(sys.argv[2], cps_password=cps_password)
            dmkcfg.upgrade_config_update(sys.argv[3], second_sensitive,
                                         sys.argv[4])
        elif sys.argv.__len__() == 6:
            cps_password_enc = six.moves.input()
            cps_password = decode_from_base64(cps_password_enc)
            dmkcfg = DmkConfig(sys.argv[2], cps_password=cps_password)
            dmkcfg.upgrade_config_update(*sys.argv[3:6])
        else:
            logger.debug(" %s value is empty,skip")

    # interface for dmk other operations
    elif sys.argv.__len__() >= 2 and sys.argv[1] in (
            'check', 'append', 'commit', 'rollback', 'sync'):
        cps_password_enc = six.moves.input()
        cps_password = decode_from_base64(cps_password_enc)
        dmkcfg = DmkConfig(sys.argv[2], cps_password=cps_password)

        # interface check, check all ebackup can connect or not
        if sys.argv.__len__() == 3 and sys.argv[1] == "check":
            dmkcfg.upgrade_config_check()

        # interface append, write key, value, section to driver_upgrade.conf
        elif sys.argv.__len__() == 6 and sys.argv[1] == "append":

            dmkcfg.upgrade_config_append(*sys.argv[3:6])

        # interface commit
        elif sys.argv.__len__() == 3 and sys.argv[1] == "commit":
            dmkcfg.upgrade_config_commit("", "")

        # interface sync
        elif sys.argv.__len__() == 3 and sys.argv[1] == "sync":
            dmkcfg.sync_config_to_all_template("", "")

        # interface rollback
        elif sys.argv.__len__() == 3 and sys.argv[1] == "rollback":
            dmkcfg.upgrade_config_rollback()

        else:
            logger.error('parameter error!')
            exit(1)
    else:
        hwconfig_main()
